{-|
Module      : Main.hs
Description : Gargantext starter
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

Script to start gargantext with different modes (Dev, Prod, Mock).

-}

{-# LANGUAGE QuasiQuotes          #-}
{-# LANGUAGE StandaloneDeriving   #-}
{-# LANGUAGE Strict               #-}
{-# LANGUAGE TypeOperators        #-}
{-# OPTIONS_GHC -fno-warn-orphans #-}
{-# LANGUAGE TypeFamilies #-}


module Main where


import Data.String (String)
import Data.Text (unpack)
import Data.Version (showVersion)
import Gargantext.API (startGargantext) -- , startGargantextMock)
import Gargantext.API.Admin.EnvTypes
import Gargantext.Prelude
import Gargantext.System.Logging
import Options.Generic
import System.Exit (exitSuccess)
import qualified Paths_gargantext as PG -- cabal magic build module


instance ParseRecord Mode
instance ParseField  Mode
instance ParseFields Mode

data MyOptions w =
  MyOptions { run  :: w ::: Mode
                        <?> "Possible modes: Dev | Mock | Prod"
            , port :: w ::: Maybe Int
                        <?> "By default: 8008"
            , ini  :: w ::: Maybe Text
                        <?> "Ini-file path of gargantext.ini"
            , version :: w ::: Bool
                        <?> "Show version number and exit"
            }
   deriving (Generic)

instance ParseRecord (MyOptions Wrapped)
deriving instance Show (MyOptions Unwrapped)

-- | A plain logger in the IO monad, waiting for more serious logging solutions like
-- the one described in https://gitlab.iscpif.fr/gargantext/haskell-gargantext/issues/229
instance HasLogger IO where
  data instance Logger IO        = IOLogger
  type instance LogInitParams IO = ()
  type instance LogPayload IO    = String
  initLogger                     = \() -> pure IOLogger
  destroyLogger                  = \_  -> pure ()
  logMsg = \IOLogger lvl msg ->
    let pfx = "[" <> show lvl <> "] "
    in putStrLn $ pfx <> msg
  logTxt lgr lvl msg = logMsg lgr lvl (unpack msg)

main :: IO ()
main = withLogger () $ \ioLogger -> do
  MyOptions myMode myPort myIniFile myVersion  <- unwrapRecord
          "Gargantext server"
  ---------------------------------------------------------------
  if myVersion then do
    logMsg ioLogger INFO $ "Version: " <> showVersion PG.version
    System.Exit.exitSuccess
  else
    return ()
  ---------------------------------------------------------------
  let myPort' = case myPort of
        Just p  -> p
        Nothing -> 8008

      myIniFile' = case myIniFile of
          Nothing -> panic "[ERROR] gargantext.ini needed"
          Just i  -> i

  ---------------------------------------------------------------
  let start = case myMode of
        Mock -> panic "[ERROR] Mock mode unsupported"
        _ -> startGargantext myMode myPort' (unpack myIniFile')
  logMsg ioLogger INFO $ "Starting with " <> show myMode <> " mode."
  start
  ---------------------------------------------------------------
