{-# LANGUAGE OverloadedStrings #-}
{-# LANGUAGE TypeApplications #-}
{-# LANGUAGE ScopedTypeVariables #-}
{-# LANGUAGE QuasiQuotes #-}

module Test.Offline.Phylo (tests) where

import Common
import Data.Aeson as JSON
import Data.Aeson.Encode.Pretty qualified as JSON
import Data.Aeson.Types qualified as JSON
import Data.ByteString.Lazy qualified as BL
import Data.GraphViz.Attributes.Complete qualified as Graphviz
import Data.Text.Encoding qualified as TE
import Data.Text.Lazy as TL
import Data.TreeDiff
import Data.Vector qualified as V
import Gargantext.Core.Text.List.Formats.CSV
import Gargantext.Core.Types.Phylo
import Gargantext.Core.Viz.Phylo
import Gargantext.Core.Viz.Phylo.API
import Gargantext.Core.Viz.Phylo.API.Tools (readPhylo, writePhylo, phylo2dot2json)
import Gargantext.Core.Viz.Phylo.Example qualified as Cleopatre
import Gargantext.Core.Viz.Phylo.PhyloExport
import Gargantext.Core.Viz.Phylo.PhyloMaker (toPhylo, toPhyloWithoutLink)
import Gargantext.Core.Viz.Phylo.PhyloTools
import Paths_gargantext
import Prelude
import Test.QuickCheck
import Test.QuickCheck.Monadic
import Test.Tasty
import Test.Tasty.HUnit
import Test.Tasty.QuickCheck

phyloConfig :: PhyloConfig
phyloConfig = PhyloConfig {
    corpusPath = "corpus.csv"
  , listPath = "list.csv"
  , outputPath = "data/"
  , corpusParser = Csv {_csv_limit = 150000}
  , listParser = V4
  , phyloName = "Phylo Name"
  , phyloScale = 2
  , similarity = WeightedLogJaccard {_wlj_sensibility = 0.5, _wlj_minSharedNgrams = 2}
  , seaElevation = Constante {_cons_start = 0.1, _cons_gap = 0.1}
  , defaultMode = True
  , findAncestors = False
  , phyloSynchrony = ByProximityThreshold {_bpt_threshold = 0.5, _bpt_sensibility = 0.0, _bpt_scope = AllBranches, _bpt_strategy = MergeAllGroups}
  , phyloQuality = Quality {_qua_granularity = 0.8, _qua_minBranch = 3}
  , timeUnit = Year {_year_period = 3, _year_step = 1, _year_matchingFrame = 5}
  , clique = MaxClique {_mcl_size = 5, _mcl_threshold = 1.0e-4, _mcl_filter = ByThreshold}
  , exportLabel = [ BranchLabel {_branch_labelTagger = MostEmergentTfIdf, _branch_labelSize = 2}
                  , GroupLabel {_group_labelTagger = MostEmergentInclusive, _group_labelSize = 2}
                  ]
  , exportSort = ByHierarchy {_sort_order = Desc}
  , exportFilter = [ByBranchSize {_branch_size = 3.0}]
  }

tests :: TestTree
tests = testGroup "Phylo" [
  testGroup "Export" [
      testCase "ngramsToLabel respects encoding" test_ngramsToLabel_01
    , testCase "ngramsToLabel is rendered correctly in CustomAttribute" test_ngramsToLabel_02
  ]
  , testGroup "toPhyloWithoutLink" [
    testCase "returns expected data" testSmallPhyloWithoutLinkExpectedOutput
  , testCase "phyloCleopatre returns expected data" testCleopatreWithoutLinkExpectedOutput
  ]
  , testGroup "phylo2dot2json" [
    testCase "is deterministic" testPhylo2dot2json
  ]
  , testGroup "toPhylo" [
    --testProperty "returns expected data" testSmallPhyloExpectedOutput
    testCase "is deterministic" testToPhyloDeterminism
  ]
  , testGroup "relatedComponents" [
    testCase "finds simple connection" testRelComp_Connected
  ]
  ]

testCleopatreWithoutLinkExpectedOutput :: Assertion
testCleopatreWithoutLinkExpectedOutput = do
  let actual = toPhyloWithoutLink Cleopatre.docs Cleopatre.config
  expected <- readPhylo =<< getDataFileName "test-data/phylo/cleopatre.golden.json"
  assertBool (show $ ansiWlEditExpr $ ediff' expected actual) (expected == actual)

testSmallPhyloWithoutLinkExpectedOutput :: Assertion
testSmallPhyloWithoutLinkExpectedOutput = do
  bpaConfig      <- getDataFileName "bench-data/phylo/bpa-config.json"
  corpusPath'    <- getDataFileName "test-data/phylo/small_phylo_docslist.csv"
  listPath'      <- getDataFileName "test-data/phylo/small_phylo_ngramslist.csv"
  (Right config) <- fmap (\pcfg -> pcfg { corpusPath = corpusPath'
                                        , listPath   = listPath'
                                        }) <$> (eitherDecodeFileStrict' bpaConfig)
  mapList <- fileToList (listParser config) (listPath config)
  corpus <- fileToDocsDefault (corpusParser config)
                        (corpusPath config)
                        [Year 3 1 5,Month 3 1 5,Week 4 2 5]
                        mapList
  actual <- phylo2dot2json (toPhyloWithoutLink corpus config)
  expected_e <- JSON.eitherDecodeFileStrict' =<< getDataFileName "test-data/phylo/small-phylo.golden.json"
  case expected_e of
    Left err -> fail err
    Right (expected :: JSON.Value) -> do
      assertBool (show $ ansiWlEditExpr $ ediff' expected actual) (expected == actual)

testPhylo2dot2json :: Assertion
testPhylo2dot2json = do
  expected_e <- JSON.eitherDecodeFileStrict' =<< getDataFileName "test-data/phylo/phylo2dot2json.golden.json"
  case expected_e of
    Left err -> fail err
    Right (jsonBlob :: JSON.Value) -> do
      actualBlob <- phylo2dot2json Cleopatre.phyloCleopatre
      let prettyConfig = JSON.defConfig { JSON.confCompare = compare }
      let actualJSON   = TE.decodeUtf8 (BL.toStrict $ JSON.encodePretty' prettyConfig $ jsonBlob)
      let expectedJSON = TE.decodeUtf8 (BL.toStrict $ JSON.encodePretty' prettyConfig $ actualBlob)
      assertBool ("JSON mismatch!" <> show (ansiWlEditExpr $ ediff' expectedJSON actualJSON)) (expectedJSON == actualJSON)

testSmallPhyloExpectedOutput :: Property
testSmallPhyloExpectedOutput = monadicIO $ do
  issue290PhyloSmall <- run $ setConfig phyloConfig <$> (readPhylo =<< getDataFileName "bench-data/phylo/issue-290-small.json")
  let actual = toPhylo issue290PhyloSmall
  expected <- run $ readPhylo =<< getDataFileName "test-data/phylo/issue-290-small.golden.json"
  pure $ counterexample (show $ ansiWlEditExpr $ ediff' expected actual) (expected === actual)

test_ngramsToLabel_01 :: Assertion
test_ngramsToLabel_01 =
  let ngrams = V.fromList [ "évaluation", "méthodologique" ]
  in ngramsToLabel ngrams [0,1] @?= "évaluation | méthodologique"

test_ngramsToLabel_02 :: Assertion
test_ngramsToLabel_02 =
  let ngrams = V.fromList [ "钱", "狗" ]
  in (Graphviz.customValue $ toAttr "lbl" $ TL.fromStrict $ ngramsToLabel ngrams [0,1]) @?= "钱 | 狗"

testRelComp_Connected :: Assertion
testRelComp_Connected = do
  (relatedComponents @Int) []                                @?= []
  (relatedComponents @Int) [[]]                              @?= [[]]
  (relatedComponents @Int) [[],[1,2]]                        @?= [[],[1,2]]
  (relatedComponents @Int) [[1,2],[]]                        @?= [[1,2],[]]
  (relatedComponents @Int) [[1,2], [2]]                      @?= [[1,2]]
  (relatedComponents @Int) [[1,2], [2],[2]]                  @?= [[1,2]]
  (relatedComponents @Int) [[1,2], [2],[2,1]]                @?= [[1,2]]
  (relatedComponents @Int) [[1,2], [2,4]]                    @?= [[1,2,4]]
  (relatedComponents @Int) [[1,2], [3,5], [2,4]]             @?= [[3,5], [1,2,4]]
  (relatedComponents @Int) [[1,2], [3,5], [2,4],[9,5],[5,4]] @?= [[1,2,4,3,5,9]]
  (relatedComponents @Int) [[1,2,5], [4,5,9]]                @?= [[1,2,5,4,9]]

testToPhyloDeterminism :: Assertion
testToPhyloDeterminism = do
  -- Acquire the config from the golden file.
  expected_e <- JSON.eitherDecodeFileStrict' =<< getDataFileName "test-data/phylo/187481.json"
  case expected_e of
    Left err -> fail err
    Right (pd :: PhyloData) -> do
      let goldenCfg  = pd_config pd
      corpusPath'    <- getDataFileName "test-data/phylo/GarganText_DocsList-nodeId-187481.csv"
      listPath'      <- getDataFileName "test-data/phylo/GarganText_NgramsList-187482.csv"
      let config     = goldenCfg { corpusPath = corpusPath'
                                 , listPath   = listPath'
                                 , listParser = V3
                                 }
      mapList <- csvMapTermList (listPath config)
      corpus <- fileToDocsDefault (corpusParser config)
                                  (corpusPath config)
                                  [Year 3 1 5,Month 3 1 5,Week 4 2 5]
                                  mapList
      let actualPhylo = toPhylo $ toPhyloWithoutLink corpus config
      actual_e   <- JSON.parseEither JSON.parseJSON <$> phylo2dot2json actualPhylo
      case actual_e of
        Left err -> fail err
        Right (expected :: GraphData) -> do
          assertBool ("Phylo mismatch! " <> show (ansiWlEditExpr $ ediff' expected (pd_data pd))) (expected == pd_data pd)
          let prettyConfig = JSON.defConfig { JSON.confCompare = compare }
          let actualJSON   = TE.decodeUtf8 (BL.toStrict $ JSON.encodePretty' prettyConfig $ pd_data pd)
          let expectedJSON = TE.decodeUtf8 (BL.toStrict $ JSON.encodePretty' prettyConfig $ expected)
          assertBool ("JSON mismatch!" <> show (ansiWlEditExpr $ ediff' expectedJSON actualJSON)) (expectedJSON == actualJSON)
