{-# LANGUAGE DeriveGeneric   #-}
{-# LANGUAGE TemplateHaskell #-}
{-# LANGUAGE TypeOperators   #-}

module Gargantext.API.Routes.Named.Table (
  -- * Routes types
    TableAPI(..)
  , TableNgramsAPI(..)
  , TableNgramsApiGet(..)
  , TableNgramsApiPut(..)
  , RecomputeScoresNgramsApiGet(..)
  , TableNgramsApiGetVersion(..)
  , TableNgramsAsyncAPI(..)

  -- * API types (appears in the routes)

  , TableQuery(..)
  , FacetTableResult
  ) where

import Data.Aeson.TH
import Data.Swagger
import Data.Text (Text)
import GHC.Generics
import Gargantext.API.Admin.Orchestrator.Types
import Gargantext.API.HashedResponse
import Gargantext.API.Ngrams.Types (TabType(..), UpdateTableNgramsCharts, Version, QueryParamR, Versioned, VersionedWithCount, NgramsTable, NgramsTablePatch)
import Gargantext.Core.Text.Corpus.Query (RawQuery)
import Gargantext.Core.Types (TableResult(..))
import Gargantext.Core.Types.Main (ListType)
import Gargantext.Core.Types.Query
import Gargantext.Core.Utils.Prefix (unPrefix, unPrefixSwagger)
import Gargantext.Database.Admin.Types.Node (ListId)
import Gargantext.Database.Query.Facet.Types
import Prelude
import Servant
import Test.QuickCheck


data TableAPI mode = TableAPI
  { getTableEp   :: mode :- Summary "Table API"
                         :> QueryParam "tabType" TabType
                         :> QueryParam "limit" Limit
                         :> QueryParam "offset" Offset
                         :> QueryParam "orderBy" OrderBy
                         :> QueryParam "query" RawQuery
                         :> QueryParam "year" Text
                         :> Get    '[JSON] (HashedResponse FacetTableResult)
  , postTableEp :: mode :- Summary "Table API (POST)"
                        :> ReqBody '[JSON] TableQuery
                        :> Post    '[JSON] FacetTableResult
  , hashTableEp :: mode :- "hash"
                        :> Summary "Hash Table"
                        :> QueryParam "tabType" TabType
                        :> Get '[JSON] Text
  } deriving Generic


data TableNgramsAPI mode = TableNgramsAPI
  { tableNgramsGetAPI       :: mode :- NamedRoutes TableNgramsApiGet
  , tableNgramsPutAPI       :: mode :- NamedRoutes TableNgramsApiPut
  , recomputeScoresEp       :: mode :- NamedRoutes RecomputeScoresNgramsApiGet
  , tableNgramsGetVersionEp :: mode :- "version" :> NamedRoutes TableNgramsApiGetVersion
  , tableNgramsAsyncAPI     :: mode :- NamedRoutes TableNgramsAsyncAPI
  } deriving Generic


data TableNgramsApiGet mode = TableNgramsApiGet
  { getNgramsTableEp :: mode :- Summary " Table Ngrams API Get"
                             :> QueryParamR "ngramsType"  TabType
                             :> QueryParamR "list"        ListId
                             :> QueryParamR "limit"       Limit
                             :> QueryParam  "offset"      Offset
                             :> QueryParam  "listType"    ListType
                             :> QueryParam  "minTermSize" MinSize
                             :> QueryParam  "maxTermSize" MaxSize
                             :> QueryParam  "orderBy"     OrderBy
                             :> QueryParam  "search"      Text
                             :> Get    '[JSON] (VersionedWithCount NgramsTable)
  } deriving Generic


data TableNgramsApiPut mode = TableNgramsApiPut
  { putNgramsTableEp :: mode :- Summary " Table Ngrams API Change"
                             :> QueryParamR "ngramsType" TabType
                             :> QueryParamR "list"       ListId
                             :> ReqBody '[JSON] (Versioned NgramsTablePatch)
                             :> Put     '[JSON] (Versioned NgramsTablePatch)
  } deriving Generic


data RecomputeScoresNgramsApiGet mode = RecomputeScoresNgramsApiGet
  { recomputeNgramsEp :: mode :- Summary " Recompute scores for ngrams table"
                              :> QueryParamR "ngramsType"  TabType
                              :> QueryParamR "list"        ListId
                              :> "recompute" :> Post '[JSON] Int
  } deriving Generic


data TableNgramsApiGetVersion mode = TableNgramsApiGetVersion
  { getTableNgramsVersion :: mode :- Summary " Table Ngrams API Get Version"
                                  :> QueryParamR "ngramsType"  TabType
                                  :> QueryParamR "list"        ListId
                                  :> Get    '[JSON] Version
  } deriving Generic


data TableNgramsAsyncAPI mode = TableNgramsAsyncAPI
  { updateTableNgramsChartsEp :: mode :- Summary "Table Ngrams Async API"
                                      :> "async"
                                      :> "charts"
                                      :> "update"
                                      :> AsyncJobs JobLog '[JSON] UpdateTableNgramsCharts JobLog
  } deriving Generic


data TableQuery = TableQuery
  { tq_offset       :: Offset
  , tq_limit        :: Limit
  , tq_orderBy      :: OrderBy
  , tq_view         :: TabType
  , tq_query        :: RawQuery
  } deriving Generic

type FacetTableResult = TableResult FacetDoc

--
-- instances
--

$(deriveJSON (unPrefix "tq_") ''TableQuery)

instance ToSchema TableQuery where
  declareNamedSchema = genericDeclareNamedSchema (unPrefixSwagger "tq_")

instance Arbitrary TableQuery where
  arbitrary = elements [TableQuery { tq_offset = 0
                                   , tq_limit = 10
                                   , tq_orderBy = DateAsc
                                   , tq_view = Docs
                                   , tq_query = "electrodes" }]
