{-|
Module      : Gargantext.Core.Text.Corpus.Query
Description : Query parsing functionality
Copyright   : (c) CNRS, 2017 - present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

-}

{-# LANGUAGE OverloadedStrings #-}
{-# LANGUAGE DerivingStrategies #-}
module Gargantext.Core.Text.Corpus.Query (
    Query -- * opaque
  , RawQuery(..)
  , Limit(..)
  , getQuery
  , parseQuery
  , mapQuery
  , renderQuery
  , interpretQuery
  , ExternalAPIs(..)
  , module BoolExpr

  -- * Useful for testing
  , unsafeMkQuery
  ) where

import Data.Aeson qualified as Aeson
import Data.BoolExpr (BoolExpr(..), Signed(..))
import Data.BoolExpr qualified as BoolExpr
import Data.BoolExpr.Parser qualified as BoolExpr
import Data.BoolExpr.Printer qualified as BoolExpr
import Data.String
import Data.Swagger qualified as Swagger
import Data.Text qualified as T
import Gargantext.Core.Text.Corpus.Types (ExternalAPIs(..))
import Gargantext.Core.Types
import Gargantext.Prelude hiding ((<|>), try)
import Servant.API qualified as Servant
import Test.QuickCheck
import Text.Parsec qualified as P
import Text.ParserCombinators.Parsec
import Text.Show (showsPrec)

-- | A raw query, as typed by the user from the frontend.
newtype RawQuery = RawQuery { getRawQuery :: T.Text }
  deriving newtype ( Show, Eq, IsString
                   , Servant.FromHttpApiData, Servant.ToHttpApiData
                   , Aeson.FromJSON, Aeson.ToJSON
                   , Swagger.ToParamSchema, Swagger.ToSchema)

instance Arbitrary RawQuery where
  arbitrary = RawQuery <$> arbitrary

-- | A limit to the number of results we want to retrieve.
newtype Limit = Limit { getLimit :: Int }
  deriving newtype ( Show, Eq, Num
                   , Servant.FromHttpApiData, Servant.ToHttpApiData
                   , Aeson.FromJSON, Aeson.ToJSON
                   , Swagger.ToParamSchema, Swagger.ToSchema)

-- | An opaque wrapper around a 'Query' type which can be parsed from a boolean
-- expression like (a AND b) OR c, and which can be interpreted in many ways
-- according to the particular service we are targeting.
newtype Query = Query { getQuery :: (BoolExpr.CNF Term) }
  deriving Show

interpretQuery :: Query -> (BoolExpr.BoolExpr Term -> ast) -> ast
interpretQuery (Query q) transform = transform . simplify . BoolExpr.fromCNF $ q

simplify :: BoolExpr.BoolExpr a -> BoolExpr.BoolExpr a
simplify expr = case expr of
  BAnd sub BTrue    -> simplify sub
  BAnd BTrue sub    -> simplify sub
  BAnd BFalse _     -> BFalse
  BAnd _ BFalse     -> BFalse
  BAnd sub1 sub2    -> BAnd (simplify sub1) (simplify sub2)
  BOr _ BTrue       -> BTrue
  BOr BTrue _       -> BTrue
  BOr sub BFalse    -> simplify sub
  BOr BFalse sub    -> simplify sub
  BOr sub1 sub2     -> BOr (simplify sub1) (simplify sub2)
  BNot BTrue        -> BFalse
  BNot BFalse       -> BTrue
  BNot (BNot sub)   -> simplify sub
  BNot sub          -> BNot (simplify sub)
  BTrue             -> BTrue
  BFalse            -> BFalse
  BConst signed     -> BConst signed

unsafeMkQuery :: BoolExpr.BoolExpr Term -> Query
unsafeMkQuery = Query . BoolExpr.boolTreeToCNF

termToken :: CharParser st Term
termToken = Term <$> (try (T.pack <$> BoolExpr.identifier) <|> (between dubQuote dubQuote multipleTerms))
  where
    dubQuote      = BoolExpr.symbol "\""
    multipleTerms = T.intercalate " " . map T.pack <$> sepBy BoolExpr.identifier BoolExpr.whiteSpace

-- | Parses an input 'Text' into a 'Query', reporting an error if it fails.
parseQuery :: RawQuery -> Either String Query
parseQuery (RawQuery txt) = bimap show (Query . BoolExpr.boolTreeToCNF) $
  P.runParser (BoolExpr.parseBoolExpr termToken) () "Corpus.Query" (T.unpack txt)

renderQuery :: Query -> RawQuery
renderQuery (Query cnf) = RawQuery . T.pack $ BoolExpr.boolExprPrinter (showsPrec 0) (BoolExpr.fromCNF cnf) ""

mapQuery :: (Term -> Term) -> Query -> Query
mapQuery f = Query . fmap f . getQuery
