{-|
Module      : Gargantext.API.Node.Corpus.New.Types
Description : New corpus API Types
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

-}

{-# LANGUAGE TemplateHaskell    #-}


module Gargantext.API.Node.Corpus.New.Types where

import Data.Aeson (genericParseJSON, genericToJSON)
import Data.Swagger (ToParamSchema, ToSchema(..), genericDeclareNamedSchema)
import Data.Text (pack)
import Gargantext.API.Node.Corpus.Types (Database, Datafield)
import Gargantext.Core (Lang(..))
import Gargantext.Core.Text.Corpus.Query qualified as Corpus
import Gargantext.Core.Text.Corpus.Parsers.Types qualified as Parsers
import Gargantext.Core.Text.List.Social (FlowSocialListWith(..))
import Gargantext.Core.Utils.Prefix (unPrefixSwagger)
import Gargantext.Prelude
import Servant (FromHttpApiData(..), ToHttpApiData(..))
import Servant.Job.Utils (jsonOptions)
import Test.QuickCheck (elements)
import Test.QuickCheck.Arbitrary (Arbitrary, arbitrary)


-- | This is just a wrapper around G.C.T.C.Parsers -> FileType for API requests
newtype FileType = FileType { unFileType :: Parsers.FileType }
  deriving (Eq, Show, Generic)
instance ToSchema FileType
instance Arbitrary FileType where arbitrary = elements [ FileType Parsers.CsvGargV3
                                                       , FileType Parsers.RisPresse]
instance ToParamSchema FileType
instance FromJSON FileType
instance ToJSON FileType

instance FromHttpApiData FileType where
  parseUrlPiece "CSV"       = pure $ FileType Parsers.CsvGargV3
  parseUrlPiece "CSV_HAL"   = pure $ FileType Parsers.CsvHal
  parseUrlPiece "Iramuteq"  = pure $ FileType Parsers.Iramuteq
  parseUrlPiece "Istex"     = pure $ FileType Parsers.Istex
  parseUrlPiece "JSON"      = pure $ FileType Parsers.JSON
  parseUrlPiece "PresseRis" = pure $ FileType Parsers.RisPresse
  parseUrlPiece "WOS"       = pure $ FileType Parsers.WOS
  parseUrlPiece s           = panicTrace $ "[G.A.A.Node.Corpus.New] File Type not implemented (yet): " <> s
instance ToHttpApiData FileType where
  toUrlPiece = pack . show



newtype FileFormat = FileFormat { unFileFormat :: Parsers.FileFormat }
  deriving (Eq, Show, Generic)
instance ToSchema FileFormat
instance Arbitrary FileFormat where arbitrary = elements [ FileFormat Parsers.Plain
                                                         , FileFormat Parsers.ZIP ]
instance ToParamSchema FileFormat
instance FromJSON FileFormat
instance ToJSON FileFormat
instance FromHttpApiData FileFormat where
  parseUrlPiece "Plain" = pure $ FileFormat Parsers.Plain
  parseUrlPiece "ZIP"   = pure $ FileFormat Parsers.ZIP
  parseUrlPiece _       = pure $ FileFormat Parsers.Plain -- TODO error here
instance ToHttpApiData FileFormat where
  toUrlPiece = pack . show


------------------------------------------------------------------------
data WithQuery = WithQuery
  { _wq_query        :: !Corpus.RawQuery
  , _wq_databases    :: !Database
  , _wq_datafield    :: !(Maybe Datafield)
  , _wq_lang         :: !Lang
  , _wq_node_id      :: !Int
  , _wq_flowListWith :: !FlowSocialListWith
  , _wq_pubmedAPIKey :: !(Maybe Text)
  , _wq_epoAPIUser   :: !(Maybe Text)
  , _wq_epoAPIToken  :: !(Maybe Text)
  }
  deriving (Show, Eq, Generic)

makeLenses ''WithQuery
instance FromJSON WithQuery where
  parseJSON = genericParseJSON $ jsonOptions "_wq_"
instance ToJSON WithQuery where
  toJSON = genericToJSON $ jsonOptions "_wq_"
instance ToSchema WithQuery where
  declareNamedSchema = genericDeclareNamedSchema (unPrefixSwagger "_wq_")

instance Arbitrary WithQuery where
  arbitrary = WithQuery <$> arbitrary
                        <*> arbitrary
                        <*> arbitrary
                        <*> arbitrary
                        <*> arbitrary
                        <*> arbitrary
                        <*> arbitrary
                        <*> arbitrary
                        <*> arbitrary

------------------------------------------------------------------------
 
