{-| Module      : Graph.Clustering
Description : 
Copyright   : (c) CNRS, Alexandre Delanoë
License     : AGPL (MIT) + CECILL (CEA/CNRS/INRIA)
Maintainer  : alexandre+dev@delanoe.org
Stability   : experimental
Portability : POSIX

Reference: Article POK de QuaC de G
POK: Parts Overlap Kern
-}

{-# LANGUAGE FlexibleContexts  #-}
{-# LANGUAGE NoImplicitPrelude #-}

module Graph.BAC.Clustering
  where

import Data.Serialize
import Control.Lens
import Data.Map (Map)
import Data.Maybe (catMaybes)
import Data.Set (Set)
import Data.Tuple (fst)
import Graph.BAC.Proxemy
import Graph.BAC.Scores
import Graph.BAC.Types
import Graph.FGL hiding (Graph)
import Protolude

import qualified Data.Serialize                    as S
import qualified Data.Graph.Inductive              as DGI
import qualified Data.Graph.Inductive.PatriciaTree as DGIP
import qualified Data.IntMap                       as IntMap
import qualified Data.List                         as List
import qualified Data.Map                          as Map
import qualified Data.Set                          as Set
-- TODO import qualified Data.Vector as Vector

---------------------------------------------------------------
-- lclusterGraph :: Length -> DGIP.Gr a Double -> Maybe (Kern (Maybe a))

{-
lclusterGraph l g = do
  k <- clusterGraph l g
  pure $ lkern g k

lclusterGraph' l g = do
  (m, ks) <- clusterGraph' l g
  pure $ lkerns m g ks

---------------------------------------------------------------
clusterGraph' :: (Monoid a, Ord a)
              => Length
              -> DGI.Gr a Double
              -> Maybe ( IntMap (Set Node)
                       , Kerns SumConf Clustering
                       )
clusterGraph' l g = do
  k <- clusterGraph l g
  let
    m  = kern2map k
    g' = toHypergraph m g
    c  = similarity_conf g' l True False
    ks = toKern c (nodes g')
  -- pure (m, kerns l g' c sumConf ks)
  pure (m, kerns l g' c qualityConf_Module ks)
-}

kern2map :: Kern score Clustering
         -> IntMap (Set Node)
kern2map (Kern s k) = IntMap.fromList
                    $ List.zip [1..] (Set.toList k)


data DataResult = Result1 ( Confluence
              , [((Node,Node), Double)]
              , Kern Score Clustering
              ) | Result2 (Kern Score Clustering)
                | Result3 (Kerns Score Clustering)
  deriving (Show)


clusterGraph :: (Serialize label, Ord label)
                => Length
                -> DGIP.Gr label Double
                -> Maybe DataResult
clusterGraph l g = do
  let c   = similarity_conf g l True False
      ks = toKerns g c (nodes g)
  k@(Kern s c') <- bestKernWith kern_score $ kerns l g c ks
  pure $ Result2 k
  -- pure $ Result1 ( c , edgesOrdered l g, k)
  -- kerns' l g c ks
  -- pure $ fusion g c kern_score k


clusterGraph_v1 :: (Serialize label, Ord label)
                => Length
                -> DGIP.Gr label Double
                -> Maybe DataResult
clusterGraph_v1 l g = do
  let c   = similarity_conf g l True False
      ks = toKerns g c (nodes g)
  pure $ Result3 $ kerns l g c ks
  -- pure $ Result1 ( c , edgesOrdered l g, k)
  -- kerns' l g c ks
  -- pure $ fusion g c kern_score k


toKerns :: ( Ord a
          , HasScore score [Set a]
          , DGI.DynGraph gr
          , Serialize label, Ord label
          )
        => gr label Double
        -> Confluence
        -> [a]
        -> Kerns score (Set (Set a))
toKerns g c ns = Kerns n [Kern n cs]
  where
    n        = hasScore g c clusters
    cs       = Set.fromList clusters
    clusters = map Set.singleton ns


toKern :: ( Ord a
          , HasScore score [Set a]
          , DGI.DynGraph gr
          , Serialize label, Ord label
          )
        => gr label Double
        -> Confluence
        -> [a]
        -> Kern score (Set (Set a))
toKern g c ns = Kern n cs
  where
    n        = hasScore g c clusters
    cs       = Set.fromList clusters
    clusters = map Set.singleton ns

------------------------------------------------------------------------
------------------------------------------------------------------------
-- | Tools for labelled kern.s
lkern :: (Ord a, DGI.Graph gr, Monoid a)
      => gr a b
      -> Kern score Clustering
      -> Kern score (Set (Set a))
lkern g (Kern s cs) =
  Kern s (Set.map (Set.map ((fromMaybe mempty) . (DGI.lab g))) cs)

lkerns :: (Ord a, DGI.Graph gr, Monoid a)
       => IntMap (Set Node)
       -> gr a b
       -> Kerns score Clustering
       -> Kerns score (Set (Set a))
lkerns m g (Kerns s cs) = Kerns s (map (lkern g) cs')
  where
    cs' = map (\(Kern s ss) -> Kern s $ toNodes m ss ) cs

    toNodes m c = Set.map (\mod -> Set.unions
                                 $ Set.map (\n -> fromMaybe mempty
                                                $ IntMap.lookup n m
                                           ) mod
                          ) c

------------------------------------------------------------------------
-- TODO more generic
-- TODO find Lens.Getter
type MyGetter t a = ((t -> Const t t) -> Kern t a -> Const t (Kern t a))

bestKernWith :: Eq score
          => MyGetter score clustering
          -> Kerns    score clustering
          -> Maybe (Kern score clustering)
bestKernWith field (Kerns s ks) = best s ks
  where
    best s [] = Nothing
    best s (k:ks) =
      if view field k == s
         then Just k
         else best s ks


kerns :: ( HasScore score Clustering
         , HasScore score Module
         , Ord score
         , Serialize label, Ord label
         )
      => Length
      -> DGIP.Gr label Double
      -> Confluence
      -> Kerns score Clustering
      -> Kerns score Clustering
kerns l g c k = foldKerns g c es k
  where
    es = map fst $ edgesOrdered l g


foldKerns :: ( DGI.DynGraph gr
             , HasScore score Clustering
             , HasScore score Module
             , Ord score
             , Serialize label, Ord label
             )
           => gr label Double
           -> Confluence
           -> [(Node, Node)]
           -> Kerns score Clustering
           -> Kerns score Clustering
foldKerns _ _ [] k = k
foldKerns g conf (e:es) (Kerns score (k:ks)) =
  foldKerns g conf es (Kerns score' (k':k:ks))
    where
      k'@(Kern s _) = kernsMerge g conf e k
      score' = if s > score then s else score

kernsMerge :: ( DGI.DynGraph gr
               , HasScore score Clustering
               , HasScore score Module
               , Ord score
               , Serialize label, Ord label
               )
           => gr label Double
           -> Confluence
           -> (Node, Node)
           -> Kern score Clustering
           -> Kern score Clustering
kernsMerge g c (n1,n2) k =
  merge g c (m1,m2) k
  where
    clusters  = view kern_data k
    clusters' = Set.toList clusters

    m1 = search n1 clusters'
    m2 = search n2 clusters'

    search :: Node -> [Module] -> Module
    search n  []     = panic "kernsMerge:search, impossible" -- <> cs $ show n
    search m0 (m:ms) =
        if Set.member m0 m
           then m
           else search m0 ms

    merge :: ( DGI.DynGraph gr
              , HasScore score clustering
              , HasScore score mod
              , HasData  mod   clustering
              , Ord score, Eq mod
              , TestScore clustering mod
              , Serialize label, Ord label
              )
          => gr label Double
          -> Confluence
          -> (mod, mod)
          -> Kern score clustering
          -> Kern score clustering
    merge g c (m1,m2) kernCurrent =
      let
        scoreCurrent = view kern_score kernCurrent
        datasCurrent = view kern_data  kernCurrent

        scoreNext = updateScore g c scoreCurrent m1 m2
        datasNext = updateData datasCurrent m1 m2
      in
        if m1 == m2
          then kernCurrent
          else if scoreNext >= scoreCurrent -- && testScore'' c datasCurrent m1 m2
            then Kern scoreNext datasNext
            else kernCurrent


---------------------------------------------------------
kerns' :: ( HasScore score Clustering
         , HasScore score Module
         , Ord score
         , Monoid (Kern score Clustering)
         , Serialize label, Ord label
         )
      => Length
      -> DGIP.Gr label Double
      -> Confluence
      -> Kern score Clustering
      -> Kern score Clustering
kerns' l g c k = fst
               $ foldl' (kernsMerge' g c) (mempty, k)
               $ map fst
               $ edgesOrdered l g


kernsMerge' :: ( DGI.DynGraph gr
               , HasScore score Clustering
               , HasScore score Module
               , Ord score
               , Serialize label, Ord label
               )
           => gr label Double
           -> Confluence
           -> (Kern score Clustering, Kern score Clustering)
           -> (Node, Node)
           -> (Kern score Clustering, Kern score Clustering)
kernsMerge' g c (bestK, k) (n1,n2) =
  merge' g c (m1,m2) (bestK,k)
  where
    clusters  = view kern_data k
    clusters' = Set.toList clusters

    m1 = search n1 clusters'
    m2 = search n2 clusters'

    search :: Node -> [Module] -> Module
    search n  []     = panic "kernsMerge:search, impossible" -- <> cs $ show n
    search m0 (m:ms) =
        if Set.member m0 m
           then m
           else search m0 ms

    merge' g c (m1,m2) (bestKern, kernCurrent) =
      let
        scoreCurrent = view kern_score kernCurrent
        datasCurrent = view kern_data  kernCurrent

        scoreNext = updateScore g c scoreCurrent m1 m2
        datasNext = updateData datasCurrent m1 m2

        nextKern = Kern scoreNext datasNext

      in
        if m1 == m2
          then (bestKern, kernCurrent)
          else if scoreNext >= scoreCurrent -- && testScore c datasCurrent m1 m2
            then (nextKern, nextKern)
            else (bestKern, nextKern)

-------------------------------------------------
-- | Misc tools
-- TODO more generic
edgesOrdered :: Length
             -> DGIP.Gr a Double
             -> [((Node,Node), Double)]
edgesOrdered l g = subSort g
                 $ List.reverse
                 $ List.sortOn snd
                 $ catMaybes
                 $ map (\(x,y)
                         -> (,) <$> Just (x,y)
                                <*> ( let prox = similarity_conf_rm g (x,y) l True
                                          val  = Map.lookup (x,y) prox
                                       in if val == Nothing
                                            then Map.lookup (y,x) prox
                                            else val
                                    )
                       )
                 $ edges g


subSort :: DGIP.Gr a Double
             -> [((Node,Node), Double)]
             -> [((Node,Node), Double)]
subSort g xs = List.concat
             $ map (List.sortOn (\((n1,n2),_) ->
                                  let (a,b) =  if n1 < n2
                                           then (n1,n2)
                                           else (n2,n1)
                                       in a * vcount + b
                                  )
                    )
             $ List.groupBy (\x y -> snd x == snd y) xs
  where
    vcount = length $ nodes g



-------------------------------------------------
fusion :: ( DGI.DynGraph gr
          , HasScore score Clustering
          , Eq score, Ord score
          , Serialize label, Ord label
          )
       => gr label Double
       -> Confluence
       -> MyGetter score Clustering
       -> Kern score Clustering
       -> Kern score Clustering
fusion conf score field k =
  let k' = tryFusion conf score field k in
    case k == k' of
      True -> k
      _    -> fusion conf score field k'

tryFusion :: ( DGI.DynGraph gr
             , HasScore score Clustering
             , Eq score, Ord score
             , Serialize label, Ord label
             )
          => gr label Double
          -> Confluence
          -> MyGetter score Clustering
          -> Kern score Clustering
          -> Kern score Clustering
tryFusion g c field cur@(Kern _ k) =
  List.maximumBy (\k1 k2 -> compare (view field k1)
                                    (view field k2)
                 ) kks
  where
    kks = cur : ks
    ks = [ let k' = Set.insert (Set.union k1 k2)
                  $ Set.delete k1 (Set.delete k2 k)
                  in Kern (hasScore g c k') k'
         | k1 <- Set.toList k
         , k2 <- Set.toList k
         , k1 > k2
         ] 

-------------------------------------------------
-------------------------------------------------
-- | Option (dépend du graph)
{-
pendants :: Graph -> (Graph, Set Node)
pendants = undefined

addPendants :: Graph -> Set Module -> Set Node -> Set Module
addPendants = undefined
-}
-------------------------------------------------
