{-| Module      : Gargantext.Core.Viz.Graph.DGI
Description : DGI main functions used in Garg
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

Main DGI funs/types to ease portability with IGraph.

-}

{-# LANGUAGE ConstrainedClassMethods #-}
{-# LANGUAGE ConstraintKinds         #-}
{-# LANGUAGE NoImplicitPrelude       #-}

module Graph.FGL where

import Protolude
import Graph.Tools (uniq)
import qualified Data.List                         as List
import qualified Data.IntMap                       as IntMap
import qualified Data.Set                          as Set
import qualified Data.Graph.Inductive              as DGI
import qualified Data.Graph.Inductive.PatriciaTree as DGIP
import qualified Data.Map                          as Map
------------------------------------------------------------------
------------------------------------------------------------------
-- | Main Types
type Node = DGI.Node -- Int
type Edge = DGI.Edge -- (Int, Int)

------------------------------------------------------------------
-- | Main Functions
mkGraph :: [Node] -> [Edge] -> DGIP.Gr () ()
mkGraph = DGI.mkUGraph

neighbors :: DGIP.Gr a b -> Node -> [Node]
neighbors = DGI.neighbors

-- TODO optimize
edges :: DGI.DynGraph gr => gr a b -> [Edge]
edges g = uniq        $ DGI.edges g


nodes :: DGIP.Gr a b -> [Node]
nodes = DGI.nodes

------------------------------------------------------------------------
-- | Graph Tools
filterNeighbors :: DGIP.Gr a b -> Node -> [Node]
filterNeighbors g n = List.nub $ neighbors g n

-- Q: why not D.G.I.deg ? (Int as result)
degree :: (DGI.DynGraph gr, Num weight) => gr a weight -> Node -> weight
degree g n = sum
           $ IntMap.elems
           $ IntMap.fromListWith (+)
           $ map swap
           $ DGI.lneighbors g n

vcount :: DGIP.Gr a b -> Double
vcount = fromIntegral . Set.size . Set.fromList . nodes


-- | TODO tests, optim and use IGraph library, fix IO ?
ecount :: DGIP.Gr a Double -> Double
ecount = sum
       . (map (\(_,_,n) -> n))
       . DGI.labEdges

------------------------------------------------------------------
-- | HyperGraph

{- TODO ?
fromHypergraph :: IntMap (Set Node) -> DGI.Gr a Int -> DGI.Gr () ()
fromHypergraph = undefined
-}
------------------------------------------------------------------------
toHypergraph ::  (Num b, Ord b, Fractional b, Monoid a, Ord a)
             => IntMap (Set Node)
             -> DGIP.Gr a b
             -> DGIP.Gr a b
toHypergraph m g = toHypergraph' m g fromIntegral


toHypergraph' :: (Num b, Ord b, Fractional b, Monoid a, Ord a)
              => IntMap (Set Node)
              -> DGIP.Gr a b
              -> (Int -> b)
              -> DGIP.Gr a b
toHypergraph' m g f = DGI.mkGraph ns es
  where
    vs = IntMap.toList m
    ns = map (\(n,ns) -> (n, List.foldl' (\m l -> m <> l) mempty
                             $ catMaybes
                             $ Set.toList
                             $ Set.map (DGI.lab g) ns
                        )
             ) vs
    es = selfNodes <> nbors
      where
        selfNodes = map (\(n,ns) -> (n, n, hasWeight g ns f)) vs
        nbors     = map (\((k1,k2), w) -> (k1,k2,w))
                  $ Map.toList
                  $ Map.mapWithKey (\(x,y) a -> if x == y then a else a/2)
                  $ Map.fromListWith (+)                                     -- TODO optim (Map (Map a))
                  $ map (\(x,y,w) -> if x < y then ((x,y),w) else ((y,x),w)) -- order edges
                  $ List.concat
                  $ map (weightedNeighbors (invIntMap m) g) vs

type ClusterId = Int
weightedNeighbors :: (Num b, Ord b)
                   => IntMap ClusterId
                   -> DGIP.Gr a b
                   -> (Node, Set Node)
                   -> [(Node,Node,b)]
weightedNeighbors m g (n,ns) = catMaybes
                             $ map (\(n',w) -> (,,) <$> Just n <*> IntMap.lookup n' m <*> Just w)
                             $ IntMap.toList
                             $ hasNeighbors g ns
  {- TODO inverse 
    where
      withEdge x y w = (,,) <$> Just n <*> IntMap.lookup n' m <*> Just w)
  -}

invIntMap :: IntMap (Set Node) -> IntMap ClusterId
invIntMap m = IntMap.fromList
            $ List.concat
            $ map swp
            $ IntMap.toList m
  where
    swp (n, ns) = List.zip (Set.toList ns) (cycle [n])
------------------------------------------------------------------------
------------------------------------------------------------------------
------------------------------------------------------------------------
class HasNeighbors a where
  hasNeighbors :: (Num b, Ord b, Ord a)
                => DGIP.Gr a' b -> a -> IntMap b

instance HasNeighbors Node where
  hasNeighbors g n = IntMap.fromListWith (+)
                    $ map swap
                    $ DGI.lneighbors g n

instance HasNeighbors (Set Node) where
  hasNeighbors g s0 = Set.foldl' (\m k -> IntMap.delete k m) (intMap g s0) s0
    where
      intMap :: (Num b, Ord b, HasNeighbors c, Ord c)
           => DGIP.Gr a b -> Set c -> IntMap b
      intMap g s = IntMap.unionsWith (+)
                 $ Set.map (hasNeighbors g) s

------------------------------------------------------------------------
-- | Tools
hasWeight :: DGIP.Gr a b -> Set Node -> (Int -> b) -> b
hasWeight g ns f = f
                 $ List.length
                 $ edges
                 $ DGI.subgraph (Set.toList ns) g

------------------------------------------------------------------------
mkGraphUfromEdges :: [(Int, Int)] -> DGIP.Gr () ()
mkGraphUfromEdges es = mkGraph ns es
  where
    ns = List.nub (a <> b)
      where
        (a, b) = List.unzip es


mkLGraphfromEdges :: [(Int, Int, Double)] -> DGIP.Gr () Double
mkLGraphfromEdges es = DGI.mkGraph ns es
  where
    ns = List.zip (List.nub (a <> b)) (List.cycle [()])
      where
        (a, b, c) = List.unzip3 es

