{-| Module      : Gargantext.Core.Viz.Graph.IGraph
Description : IGraph main functions used in Garg
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

Main IGraph funs/types to ease portability with FGL.

Reference:
* Gábor Csárdi, Tamás Nepusz: The igraph software package for complex network research. InterJournal Complex Systems, 1695, 2006.

-}

{-# LANGUAGE NoImplicitPrelude       #-}

module Graph.IGraph where

import Data.Serialize (Serialize)
import Data.Singletons (SingI)
import IGraph hiding (mkGraph, neighbors, edges, nodes, Node, Graph)
import Protolude
import qualified Data.Graph.Inductive        as DGI
import qualified Data.List                   as List
import qualified IGraph                      as IG
import qualified IGraph.Algorithms.Clique    as IG
import qualified IGraph.Algorithms.Community as IG
import qualified IGraph.Random               as IG

------------------------------------------------------------------
-- | Main Types
type Graph_Undirected = IG.Graph 'U () ()
type Graph_Directed   = IG.Graph 'D () ()

type Node = IG.Node
type Graph = IG.Graph

------------------------------------------------------------------
-- | Main Graph management Functions
ineighbors :: IG.Graph d v e -> IG.Node -> [Node]
ineighbors = IG.neighbors

iedges :: IG.Graph d v e -> [Edge]
iedges = IG.edges

inodes :: IG.Graph d v e -> [Node]
inodes = IG.nodes

------------------------------------------------------------------
------------------------------------------------------------------
------------------------------------------------------------------
type Seed = Int

-- | Tools to analyze graphs
clusteringWith_spinGlass :: (Serialize v, Serialize e)
                         => Seed -> IG.Graph 'U v e -> IO [[Int]]
clusteringWith_spinGlass s g = do
  gen <- IG.withSeed s pure
  IG.findCommunity g Nothing Nothing IG.spinglass gen


maximalCliques :: IG.Graph d v e -> [[Int]]
maximalCliques g = IG.maximalCliques g (min',max')
  where
    min' = 0
    max' = 0

------------------------------------------------------------------
-- | Main sugared functions to make/import graphs
iGraphmkGraph :: (SingI d, Ord v,
             Serialize v, Serialize e) =>
     [v] -> [LEdge e] -> IG.Graph d v e
iGraphmkGraph = IG.mkGraph


fromFGL :: ( DGI.DynGraph gr
           , Serialize a
           , Serialize b
           , Ord       a
           )
        => gr a b
        -> IG.Graph 'U DGI.Node b
fromFGL g = iGraphmkGraph ns es
  where
    ns = DGI.nodes g
    es = map (\(n1,n2,w) -> ((n1,n2),w)) $ DGI.labEdges g


mkGraphUfromEdges_iGraph :: [(Int, Int)] -> Graph_Undirected
mkGraphUfromEdges_iGraph es = iGraphmkGraph (List.replicate n ()) $ zip es $ repeat ()
  where
    (a,b) = List.unzip es
    n = List.length (List.nub $ a <> b)

{-
mkGraphDfromEdges_iGraph :: [(Int, Int)] -> Graph_Directed
mkGraphDfromEdges_iGraph = undefined
-}
