{-| Module      : Graph.Tools
Description :
Copyright   : (c) CNRS, Alexandre Delanoë
License     : AGPL + CECILL v3
Maintainer  : alexandre+dev@delanoe.org
Stability   : experimental
Portability : POSIX


-}

{-# LANGUAGE ConstraintKinds #-}
{-# LANGUAGE InstanceSigs    #-}
{-# LANGUAGE NoImplicitPrelude       #-}


module Graph.Tools where

import Data.Csv
import Data.Text (pack, splitOn, unpack)
import Data.Vector hiding (map, uniq)
import Prelude (read)
import Protolude
import qualified Data.Map                          as Map
import qualified Data.ByteString.Lazy              as BL
import qualified Data.Graph.Inductive              as DGI
import qualified Data.Graph.Inductive.PatriciaTree as DGIP
import qualified Data.List                         as List
import qualified Data.Set                          as Set
import qualified Data.Vector                       as Vector

------------------------------------------------------------------------
uniq :: [DGI.Edge] -> [DGI.Edge]
uniq ns = Set.toList
        $ Set.fromList
        $ map (\(n1,n2)
                -> if n1 <= n2
                      then (n1,n2)
                      else (n2,n1)
              ) ns


uniq' :: [(Int,Int,Double)] -> [(Int,Int,Double)]
uniq' ns = map (\(n1,n2) -> (n1,n2,1))
         $ Map.keys
         $ Map.fromListWith (+)
         $ map (\(n1,n2,w)
                  -> if n1 <= n2
                        then ((n1,n2), w)
                        else ((n2,n1), w)
                ) ns

------------------------------------------------------------------------

data FileGraph = CillexGraph | TestGraph

readFileGraph :: FileGraph -> FilePath -> IO (DGIP.Gr [Text] Double)
readFileGraph CillexGraph fp = toGraph <$> snd <$> readFileCsvCillex fp
readFileGraph TestGraph   fp = toGraph <$> snd <$> readFileCsvTest   fp


toGraph :: ToNode a => Vector a -> (DGIP.Gr [Text] Double)
toGraph vs = DGI.mkGraph ns (uniq' $ List.concat es)
  where
    (ns,es) = List.unzip
            $ Vector.toList
            $ Vector.map toNode vs

------------------------------------------------------------------------
class ToNode a where
  toNode :: a -> ((Int, [Text]), [(Int,Int,Double)])

instance ToNode CillexCsv
  where
    toNode :: CillexCsv -> ((Int, [Text]), [(Int,Int,Double)])
    toNode (CillexCsv n v _ _ l) = (ln, es)
      where
        ln = (n, [l])
        es = List.zip3 (cycle [n]) ns (cycle [1])
        ns = map (read . unpack) $ splitOn "," v

instance ToNode TestCsv
  where
    toNode :: TestCsv -> ((Int, [Text]), [(Int,Int,Double)])
    toNode (TestCsv n v l) = (ln, es)
      where
        ln = (n, map show [l])
        es = List.zip3 (cycle [n]) ns (cycle [1])
        ns = map (read . unpack) $ splitOn "," v

------------------------------------------------------------------------
readFileCsvCillex :: FilePath -> IO (Header, Vector CillexCsv)
readFileCsvCillex = fmap readWith . BL.readFile

readFileCsvTest :: FilePath -> IO (Header, Vector TestCsv)
readFileCsvTest = fmap readWith . BL.readFile

csvDecodeOptions :: DecodeOptions
csvDecodeOptions = defaultDecodeOptions {decDelimiter = delimiter}
  where
    delimiter :: Word8
    delimiter = fromIntegral $ ord ','

readWith :: FromNamedRecord a => BL.ByteString -> (Header, Vector a)
readWith bs = case decodeByNameWith csvDecodeOptions bs of
      Left e        -> panic (pack e)
      Right csvDocs -> csvDocs

------------------------------------------------------------------------
data TestCsv =
  TestCsv { rc_node    :: Int
          , rc_voisins :: Text
          , rc_id      :: Double
          }
    deriving (Show)

instance FromNamedRecord TestCsv where
  parseNamedRecord r = TestCsv <$> r .: "@node: #id_pdg"
                               <*> r .: "%+voisins"
                               <*> r .: "label"
------------------------------------------------------------------------
data CillexCsv =
  CillexCsv { node    :: Int
            , voisins :: Text
            , id      :: Double
            , lemme   :: Text
            , label   :: Text
            }
    deriving (Show)

instance FromNamedRecord CillexCsv where
  parseNamedRecord r = CillexCsv <$> r .: "@node: #id_pdg"
                                 <*> r .: "%+voisins"
                                 <*> r .: "id"
                                 <*> r .: "lemme"
                                 <*> r .: "label"

{-
cillexCsvHeader =
  header [ "@node: #id_pdg"
         , "%+voisins"
         , "id"
         , "lemme"
         , "label"
         ]

-}

