{-| Module  : Data.Eigen.Coeff
Description : Safe coeff for Eigen Matrix
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

Author: Alp Mestanogullari (Well Typed)

-}

{-# LANGUAGE DataKinds, GADTs, RankNTypes, ScopedTypeVariables #-}
{-# LANGUAGE TypeApplications, TypeOperators, TypeFamilies #-}

module Data.Eigen.Coeff (coeffSM, coeffDM)
  where

import Prelude hiding (pi)

import Eigen.Internal (Row(..), Col(..), Elem)
import Data.Maybe (fromMaybe)
import Data.Proxy
import Data.Reflection
import Data.Type.Equality
import GHC.TypeLits
import Unsafe.Coerce
import Eigen.SparseMatrix (SparseMatrix)
import Eigen.Matrix (Matrix)

import qualified Eigen.SparseMatrix as SMatrix
import qualified Eigen.Matrix       as DMatrix

data (:<=?) (i :: Nat) (n :: Nat) where
  IsLTE :: (i <=? n) :~: 'True  -> i :<=? n
  IsGT  ::                         i :<=? n

isLTE :: (KnownNat i, KnownNat n) => Proxy i -> Proxy n -> i :<=? n
isLTE pi pn
  | natVal pi <= natVal pn = IsLTE (unsafeCoerce Refl)
  | otherwise              = IsGT

reifyNatLT
  :: forall r n. KnownNat n
  => Int
  -> Proxy n
  -> r -- ^ what to return if the given 'Int' is > n
  -> (forall i. (KnownNat i, i <= n) => Proxy i -> r)
     -- ^ what to do if the given 'Int' indeed is <= n
  -> r
reifyNatLT i pn r k = reifyNat (fromIntegral i) $ \pi ->
  case isLTE pi pn of
    IsLTE Refl -> k pi
    _          -> r

-- * matrix use case

{-
-- say we're given those things (origin Eigen lib)
data Mat (n :: Nat) (m :: Nat) a
data Row (i :: Nat) = Row
data Col (j :: Nat) = Col
coeff
  :: (KnownNat i, KnownNat j, KnownNat n, KnownNat m, i <= n, j <= m)
  => Row i -> Col j -> Mat n m a -> a
coeff = SMatrix.coeff
-}

-- we can now get:

coeffSM
  :: forall n m.
     (KnownNat n, KnownNat m)
  => Int -> Int -> SparseMatrix n m Double -> Double
coeffSM x y m = fromMaybe 0 $ coeffSM' x y m

coeffSM'
  :: forall n m a.
     (KnownNat n, KnownNat m, Elem a)
  => Int -> Int -> SparseMatrix n m a -> Maybe a
coeffSM' i j mat =
  reifyNatLT i pn Nothing $ \pi ->
  reifyNatLT j pm Nothing $ \pj ->
    Just $ SMatrix.coeff (proxyRow pi) (proxyCol pj) mat

  where pn = Proxy @n
        pm = Proxy @m

coeffDM
  :: forall n m.
     (KnownNat n, KnownNat m)
  => Int -> Int -> Matrix n m Double -> Double
coeffDM x y m = fromMaybe 0 $ coeffDM' x y m


coeffDM'
  :: forall n m a.
     (KnownNat n, KnownNat m, Elem a)
  => Int -> Int -> Matrix n m a -> Maybe a
coeffDM' i j mat =
  reifyNatLT i pn Nothing $ \pi ->
  reifyNatLT j pm Nothing $ \pj ->
    Just $ DMatrix.coeff (proxyRow pi) (proxyCol pj) mat

  where pn = Proxy @n
        pm = Proxy @m




-- helpers
proxyRow :: Proxy i -> Row i
proxyRow _ = Row
proxyCol :: Proxy j -> Col j
proxyCol _ = Col
