{-|
Module      : Gargantext.Database.Flow.Types
Description : Types for Flow
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

-}

{-# LANGUAGE ConstraintKinds         #-}
{-# LANGUAGE ConstrainedClassMethods #-}
{-# LANGUAGE InstanceSigs            #-}
{-# LANGUAGE TemplateHaskell         #-}
{-# LANGUAGE LambdaCase #-}
{-# LANGUAGE DerivingStrategies #-}

module Gargantext.Database.Action.Flow.Types
    where

import Conduit (ConduitT)
import Data.HashMap.Strict (HashMap)
import Data.Swagger (ToSchema(..), genericDeclareNamedSchema)
import Gargantext.Core.Flow.Types ( UniqId )
import Gargantext.Core.NodeStory.Types ( HasNodeStory )
import Gargantext.Core.Text ( HasText )
import Gargantext.API.Admin.Orchestrator.Types qualified as API
import Gargantext.Core.Text.Ngrams (NgramsType(..))
import Gargantext.Core.Types (HasValidationError, TermsCount, TermsWeight)
import Gargantext.Core.Utils.Prefix (unPrefix, unPrefixSwagger)
import Gargantext.Database.Admin.Types.Hyperdata.Document ( HyperdataDocument )
import Gargantext.Database.Admin.Types.Node (NodeId)
import Gargantext.Database.Prelude (IsDBCmdExtra)
import Gargantext.Database.Query.Table.Node.Document.Insert ( UniqParameters, InsertDb, ToNode, AddUniqId )
import Gargantext.Database.Query.Table.Node.Error (HasNodeError)
import Gargantext.Database.Query.Tree.Error (HasTreeError)
import Gargantext.Database.Types (Indexed)
import Gargantext.Prelude
import Gargantext.System.Logging ( MonadLogger )
import Gargantext.Core.Notifications.CentralExchange.Types (HasCentralExchangeNotification)
import Gargantext.Core.Worker.Jobs.Types (Job)
import Data.Conduit (transPipe)


type FlowCmdM env err m =
  ( IsDBCmdExtra   env err m
  , HasNodeStory   env err m
  , HasNodeError       err
  , HasValidationError err
  , HasTreeError       err
  , MonadLogger            m
  , HasCentralExchangeNotification env
  )

type FlowCorpus a = ( UniqParameters a
                    , InsertDb         a
                    , HasText          a
                    , ToNode           a
                    , ToJSON           a
                    )

type FlowInsertDB a = ( AddUniqId a
                      , UniqId    a
                      , UniqParameters a
                      , InsertDb  a
                      )



data DocumentIdWithNgrams ix a b =
     DocumentIdWithNgrams
     { documentWithId :: Indexed ix a
     , documentNgrams :: HashMap b (Map NgramsType TermsWeight, TermsCount)
     } deriving (Show)


-- TODO use internal with API name (could be old data)
data DataOrigin = InternalOrigin { _do_api :: API.ExternalAPIs }
                | ExternalOrigin { _do_api :: API.ExternalAPIs }
               -- TODO Web
  deriving (Generic, Eq)

makeLenses ''DataOrigin
deriveJSON (unPrefix "_do_") ''DataOrigin
instance ToSchema DataOrigin where
  declareNamedSchema = genericDeclareNamedSchema (unPrefixSwagger "_do_")

data DataProducer m a
  = -- | Produces documents in batches, and as such the inner 'IO' action must return
  -- a small number of documents.
    DataBatchProducer [m [a]]
  -- | Produces documents in a streaming fashion, and as such it's well suited for
  -- tasks which cannot be easily parallelised (for example external APIs returning tokens
  -- to strep through the pagination without given access to ways to randomly jump to the
  -- desired page of results).
  | DataStreamingProducer (ConduitT () a m ())
  -- | A data producer that knows how to generate jobs out of its task.
  | DataAsyncBatchProducer  [Job]

hoistDataProducer :: Monad m => (forall x. m x -> n x) -> DataProducer m a -> DataProducer n a
hoistDataProducer hoistFn = \case
  DataBatchProducer batches -> DataBatchProducer $ map hoistFn batches
  DataStreamingProducer conduitData -> DataStreamingProducer (transPipe hoistFn conduitData)
  DataAsyncBatchProducer jobs -> DataAsyncBatchProducer jobs

newtype ResultsCount = ResultsCount { _ResultsCount :: Integer }
  deriving newtype (Show, Ord, Eq)

data DataText m =
    -- | We found some old (cached) data we can serve directly
    DataOld ![NodeId]
    -- | We need to produce the new data
  | DataNew !ResultsCount !(DataProducer m HyperdataDocument)

hoistDataText :: Monad m => (forall x. m x -> n x) -> DataText m -> DataText n
hoistDataText hoistFn = \case
  DataOld old -> DataOld old
  DataNew res producer -> DataNew res (hoistDataProducer hoistFn producer)
