{-|
Module      : Gargantext.Core.AsyncUpdates.CentralExchange
Description : Central exchange (asynchronous notifications)
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

https://gitlab.iscpif.fr/gargantext/haskell-gargantext/issues/341

Docs:
https://dev.sub.gargantext.org/#/share/Notes/187918
    
-}

module Gargantext.Core.AsyncUpdates.CentralExchange where

-- import Control.Concurrent (threadDelay)
import Control.Concurrent.Async qualified as Async
import Control.Concurrent.STM.TChan qualified as TChan
import Data.Aeson qualified as Aeson
import Data.ByteString.Char8 qualified as C
import Data.ByteString.Lazy qualified as BSL
import Gargantext.Core.AsyncUpdates.CentralExchange.Types
import Gargantext.Core.AsyncUpdates.Constants (cePort, dispatcherInternalPort)
-- import Gargantext.Core.AsyncUpdates.Nanomsg (withSafeSocket)
import Gargantext.Prelude
import Nanomsg (Pull(..), Push(..), bind, connect, recvMalloc, send, withSocket)
    
{-

Central exchange is a service, which gathers messages from various
places and informs the Dispatcher (which will then inform users about
various events).

The primary goal is to be able to read as many messages as possible
and then send them to the Dispatcher. Although nanomsg does some
message buffering, we don't want these messages to pile up, especially
with many users having updates.
    
-}

gServer :: IO ()
gServer = do
  withSocket Pull $ \s -> do
    withSocket Push $ \s_dispatcher -> do
      _ <- bind s ("tcp://*:" <> show cePort)
      _ <- connect s_dispatcher ("tcp://localhost:" <> show dispatcherInternalPort)

      tChan <- TChan.newTChanIO

      -- | We have 2 threads: one that listens for nanomsg messages
      -- | and puts them on the 'tChan' and the second one that reads
      -- | the 'tChan' and calls Dispatcher accordingly. This is to
      -- | make reading nanomsg as fast as possible.
      void $ Async.concurrently (worker s_dispatcher tChan) $ do
        forever $ do
          -- putText "[central_exchange] receiving"
          r <- recvMalloc s 1024
          C.putStrLn $ "[central_exchange] " <> r
          atomically $ TChan.writeTChan tChan r
  where
    worker s_dispatcher tChan = do
      forever $ do
        r <- atomically $ TChan.readTChan tChan
        case Aeson.decode (BSL.fromStrict r) of
          Just ujp@(UpdateJobProgress _jId _jobLog) -> do
            putText $ "[central_exchange] " <> show ujp
            -- send the same message that we received
            send s_dispatcher r
          Just (UpdateTreeFirstLevel node_id) -> do
            putText $ "[central_exchange] update tree: " <> show node_id
            -- putText $ "[central_exchange] sending that to the dispatcher: " <> show node_id
            -- To make this more robust, use withAsync so we don't
            -- block the main thread (send is blocking)
            
            -- NOTE: If we're flooded with messages, and send is
            -- slow, we might be spawning many threads...
  
            -- NOTE: Currently we just forward the message that we
            -- got. So in theory central exchange isn't needed (we
            -- could ping dispatcher directly). However, I think
            -- it's better to have this as a separate
            -- component. Currently I built this inside
            -- gargantext-server but maybe it can be a separate
            -- process, independent of the server.
            -- send the same message that we received
            send s_dispatcher r
          _ -> putText $ "[central_exchange] unknown message"
      

notify :: CEMessage -> IO ()
notify ceMessage = do
  Async.withAsync (pure ()) $ \_ -> do
    withSocket Push $ \s -> do
      _ <- connect s ("tcp://localhost:" <> show cePort)
      let str = Aeson.encode ceMessage
      send s $ BSL.toStrict str
