{-# LANGUAGE TemplateHaskell #-}
module Gargantext.API.Admin.Settings.TOML where

import Control.Lens hiding ((.=))
import Data.Text qualified as T
import Gargantext.API.Admin.Settings.CORS
import Gargantext.API.Admin.Settings.MicroServices
import Gargantext.Prelude (panicTrace)
import Gargantext.System.Logging
import Paths_gargantext
import Prelude
import Toml

-- | Compatibility bridge until we fix #304 (move to Toml)
data GargTomlSettings = GargTomlSettings
  { _gargCorsSettings          :: !CORSSettings
  , _gargMicroServicesSettings :: !MicroServicesSettings
  }

makeLenses ''GargTomlSettings

settingsCodec :: TomlCodec GargTomlSettings
settingsCodec = GargTomlSettings
  <$> (Toml.table corsSettingsCodec "cors" .= _gargCorsSettings)
  <*> (Toml.table  microServicesSettingsCodec "microservices" .= _gargMicroServicesSettings)

-- | Loads the 'CORSSettings' from the 'toml' file.
loadGargTomlSettings :: IO GargTomlSettings
loadGargTomlSettings = do
  tomlFile <- getDataFileName "gargantext-settings.toml"
  tomlRes <- Toml.decodeFileEither settingsCodec tomlFile
  case tomlRes of
      Left errs      -> do
        withLogger () $ \ioLogger -> do
          logMsg ioLogger ERROR $ T.unpack $ "Error, gargantext-settings.toml parsing failed: " <> Toml.prettyTomlDecodeErrors errs
          panicTrace "Please fix the errors in your gargantext-settings.toml file."
      Right settings0 -> case settings0 ^. gargCorsSettings . corsUseOriginsForHosts of
        True  -> pure $ settings0 & over (gargCorsSettings . corsAllowedHosts) (\_ -> "http://localhost:3000" : (settings0 ^. gargCorsSettings . corsAllowedOrigins))
        False -> pure $ settings0 & over (gargCorsSettings . corsAllowedHosts) ("http://localhost:3000" :)
