{-|
Module      : Gargantext.Core.Config.Ini.Mail
Description :
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

-}

module Gargantext.Core.Config.Ini.Mail (
  -- * Types
    GargMail(..)
  , LoginType(..)
  , MailConfig(..)

  -- * Utility functions
  , gargMail
  , readConfig
  )
  where

import Data.Maybe
import Data.Text (unpack)
import Gargantext.Core.Config.Ini.Ini (readIniFile', val)
import Gargantext.Core.Config.Mail (LoginType(..), MailConfig(..), SendEmailType(LogEmailToConsole))
import Gargantext.Prelude
import Network.Mail.Mime (plainPart)
import Network.Mail.SMTP hiding (htmlPart, STARTTLS)
import Prelude (read)


type Email = Text
type Name  = Text

readConfig :: FilePath -> IO MailConfig
readConfig fp = do
  ini <- readIniFile' fp
  let val' = val ini "mail"

  pure $ MailConfig { _mc_mail_host       =        cs $ val' "MAIL_HOST"
                    , _mc_mail_port       = read $ cs $ val' "MAIL_PORT"
                    , _mc_mail_user       =        cs $ val' "MAIL_USER"
                    , _mc_mail_from       =        cs $ val' "MAIL_FROM"
                    , _mc_mail_password   =        cs $ val' "MAIL_PASSWORD"
                    , _mc_mail_login_type = read $ cs $ val' "MAIL_LOGIN_TYPE"
                    , _mc_send_login_emails = LogEmailToConsole
                    }


data GargMail = GargMail { gm_to      :: Email
                         , gm_name    :: Maybe Name
                         , gm_subject :: Text
                         , gm_body    :: Text
                         }

-- | TODO add parameters to gargantext.ini
gargMail :: MailConfig -> GargMail -> IO ()
gargMail (MailConfig {..}) (GargMail { .. }) = do
  let host     = unpack _mc_mail_host
      user     = unpack _mc_mail_user
      password = unpack _mc_mail_password
  case _mc_mail_login_type of
    NoAuth   -> sendMail                   host mail
    Normal   -> sendMailWithLogin'         host _mc_mail_port user password mail
    SSL      -> sendMailWithLoginTLS'      host _mc_mail_port user password mail
    TLS      -> sendMailWithLoginTLS'      host _mc_mail_port user password mail
    STARTTLS -> sendMailWithLoginSTARTTLS' host _mc_mail_port user password mail
  where
    mail = simpleMail sender receiver cc bcc gm_subject [plainPart $ cs gm_body]

    sender   = Address (Just "GarganText Email") _mc_mail_from
    receiver = [Address gm_name gm_to]
    cc       = []
    bcc      = []

