{-|
Module      : Gargantext.Core.Text.Corpus.Parsers.TSV.TSVv3
Description :
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

-}

module Gargantext.Core.Text.Corpus.Parsers.TSV.TSVv3 where

import Data.Csv
import Data.Text (pack)
import Data.Time.Segment (jour)
import Gargantext.Core.Text.Corpus.Parsers.TSV.Diagnostics
import Gargantext.Core.Text.Corpus.Parsers.TSV.IntOrDec
import Gargantext.Database.Admin.Types.Hyperdata.Document ( HyperdataDocument(..) )
import Gargantext.Prelude


data TsvGargV3 = TsvGargV3
    { d_docId             :: !Int
    , d_title             :: !Text
    , d_source            :: !Text
    , d_publication_year  :: !Int
    , d_publication_month :: !Int
    , d_publication_day   :: !Int
    , d_abstract          :: !Text
    , d_authors           :: !Text
    }
    deriving (Show)



data TsvDoc = TsvDoc
    { tsv_title             :: !Text
    , tsv_source            :: !Text
    , tsv_publication_year  :: !(Maybe IntOrDec)
    , tsv_publication_month :: !(Maybe Int)
    , tsv_publication_day   :: !(Maybe Int)
    , tsv_abstract          :: !Text
    , tsv_authors           :: !Text
    }
    deriving (Show)


tsv2doc :: TsvDoc -> HyperdataDocument
tsv2doc (TsvDoc { .. })
  = HyperdataDocument { _hd_bdd = Just "TsvHal"
                      , _hd_doi = Nothing
                      , _hd_url = Nothing
                      , _hd_page = Nothing
                      , _hd_title = Just tsv_title
                      , _hd_authors = Just tsv_authors
                      , _hd_institutes = Nothing
                      , _hd_source = Just tsv_source
                      , _hd_abstract = Just tsv_abstract
                      , _hd_publication_date = Just $ pack . show $ jour (fromIntegral pubYear)
                                                                         pubMonth
                                                                         pubDay
                      , _hd_publication_year = Just pubYear
                      , _hd_publication_month = Just pubMonth
                      , _hd_publication_day = Just pubDay
                      , _hd_publication_hour = Nothing
                      , _hd_publication_minute = Nothing
                      , _hd_publication_second = Nothing
                      , _hd_language_iso2 = Nothing
                      , _hd_institutes_tree = Nothing }
  where
    pubYear = fromMIntOrDec defaultYear tsv_publication_year
    pubMonth = fromMaybe defaultMonth tsv_publication_month
    pubDay = fromMaybe defaultDay tsv_publication_day



instance FromNamedRecord TsvDoc where
  parseNamedRecord r = do
    tsv_title <- r .: "title" <|> r .: "Title"
    tsv_source <- r .: "source" <|> r .: "Source"
    tsv_publication_year <- r .: "publication_year" <|> r .: "Publication Year"
    tsv_publication_month <- r .: "publication_month" <|> r .: "Publication Month"
    tsv_publication_day <- r .: "publication_day" <|> r .: "Publication Day"
    tsv_abstract <- r .: "abstract" <|> r .: "Abstract"
    tsv_authors <- r .: "authors" <|> r .: "Authors"
    pure $ TsvDoc { .. }

instance DefaultOrdered TsvDoc where
  headerOrder _ = header [ "Publication Day"
                         , "Publication Month"
                         , "Publication Year"
                         , "Authors"
                         , "Title"
                         , "Source"
                         , "Abstract"]

instance ToNamedRecord TsvDoc where
  toNamedRecord (TsvDoc{ .. }) =
    namedRecord [ "Title"             .= tsv_title
                , "Source"            .= tsv_source
                , "Publication Year"  .= tsv_publication_year
                , "Publication Month" .= tsv_publication_month
                , "Publication Day"   .= tsv_publication_day
                , "Abstract"          .= tsv_abstract
                , "Authors"           .= tsv_authors
                ]


doc2tsv :: HyperdataDocument -> TsvDoc
doc2tsv h = TsvDoc { tsv_title = m $ _hd_title h
                   , tsv_source = m $ _hd_source h
                   , tsv_publication_year = Just $ IntOrDec $ mI $ _hd_publication_year h
                   , tsv_publication_month = Just $ mI $ _hd_publication_month h
                   , tsv_publication_day = Just $ mI $ _hd_publication_day   h
                   , tsv_abstract = m $ _hd_abstract h
                   , tsv_authors = m $ _hd_authors h }

  where
    m = maybe "" identity
    mI = maybe 0 identity


validators :: [ColumnValidator]
validators =
  [ ("Publication Day", validNumber)
  , ("Publication Month", validNumber)
  , ("Publication Year", validNumber)
  , ("Authors", validTextField)
  , ("Title", validTextField)
  , ("Source", validTextField)
  , ("Abstract", validTextField)
  ]
