{-|
Module      : Gargantext.Core.Worker.Jobs
Description : Worker job definitions
Copyright   : (c) CNRS, 2024
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

-}


module Gargantext.Core.Worker.Jobs where


import Async.Worker qualified as W
import Control.Lens (view)
import Gargantext.Core.Config (gc_database_config, gc_worker, HasConfig(..), GargConfig)
import Gargantext.Core.Config.Worker (WorkerSettings(..), WorkerDefinition(..))
import Gargantext.Core.Worker.Broker (initBrokerWithDBCreate)
import Gargantext.Core.Worker.Jobs.Types (Job(..))
import Gargantext.Core.Worker.PGMQTypes (HasWorkerBroker, MessageId, SendJob)
import Gargantext.Database.Prelude (Cmd')
import Gargantext.Prelude
import Gargantext.System.Logging (logMsg, withLogger, LogLevel(..))


sendJob :: (HasWorkerBroker, HasConfig env)
        => Job
        -> Cmd' env err MessageId
sendJob job = do
  gcConfig <- view $ hasConfig
  liftBase $ sendJobCfg gcConfig job

sendJobCfg :: GargConfig -> Job -> IO MessageId
sendJobCfg gcConfig job = do
  let ws@WorkerSettings { _wsDefinitions, _wsDefaultDelay } = gcConfig ^. gc_worker
  -- TODO Try to guess which worker should get this job
  -- let mWd = findDefinitionByName ws workerName
  let mWd = head _wsDefinitions
  case mWd of
    Nothing -> panicTrace "No worker definitions available"
    Just wd -> do
      b <- initBrokerWithDBCreate (gcConfig ^. gc_database_config) ws
      let queueName = _wdQueue wd
      let job' = (updateJobData job $ W.mkDefaultSendJob' b queueName job) { W.delay = _wsDefaultDelay }
      withLogger () $ \ioL ->
        logMsg ioL DEBUG $ "[sendJob] sending job " <> show job <> " (delay " <> show (W.delay job') <> ")"
      W.sendJob' job'

-- | We want to fine-tune job metadata parameters, for each job type
updateJobData :: Job -> SendJob -> SendJob
updateJobData (AddCorpusFormAsync {}) sj = sj { W.timeout = 300 }
updateJobData (AddCorpusWithQuery {}) sj = sj { W.timeout = 3000 }
updateJobData _ sj = sj { W.resendOnKill = False }
