# https://nix.dev/tutorials/first-steps/towards-reproducibility-pinning-nixpkgs.html
{ pkgs ? import
    (if builtins.elem builtins.currentSystem [ "x86_64-darwin" "aarch64-darwin" ]
    then ./pinned-25.05.darwin.nix
    else ./pinned-25.05.nix)
    { }
}:

rec {
  inherit pkgs;

  # Create a custom stdenv with GCC 14 on Linux. This is because
  # pkgs.stdenv is GCC version 13 and when using 'nix' inside the
  # shell, one gets strange 'CXX_ABI' errors.
  customStdenv =
    if pkgs.stdenv.isLinux then
      pkgs.gcc14Stdenv
    else
      pkgs.stdenv; # Keep default on other platforms (clang on macOS)

  ghcVersion = "ghc966";
  gargGhc = pkgs.haskell.compiler.${ghcVersion};
  cabal_install = pkgs.haskell.lib.compose.justStaticExecutables pkgs.haskell.packages.${ghcVersion}.cabal-install;

  graphviz = pkgs.callPackage ./graphviz.nix { };
  igraph_0_10_4 = pkgs.callPackage ./igraph.nix { };
  cabal2stack = pkgs.callPackage ./cabal2stack.nix { ghc = gargGhc; };
  nng_notls = pkgs.nng.overrideAttrs (old: {
    cmakeFlags = (old.cmakeFlags or [ ]) ++ [ "-DNNG_ENABLE_TLS=OFF" ];
  });

  hsBuildInputs = [
    gargGhc
    cabal_install
    pkgs.haskellPackages.alex
    pkgs.haskellPackages.happy
    pkgs.haskellPackages.pretty-show
  ];
  nonhsBuildInputs = with pkgs; [
    #haskell-language-server
    blas
    bzip2
    cabal2stack
    curl
    czmq
    docker-compose
    expat
    gfortran
    git
    gmp
    graphviz
    gsl
    hlint
    icu
    igraph_0_10_4
    jre
    lapack
    libffi
    libpqxx
    libsodium
    nng_notls
    nil # nix language server
    openblas
    pcre
    pkg-config
    postgresql
    customStdenv.cc.cc
    xz
    zlib
    zeromq
    curl
  ] ++ (lib.optionals stdenv.isDarwin [
    darwin.apple_sdk.frameworks.Accelerate
  ]);
  libPaths = pkgs.lib.makeLibraryPath nonhsBuildInputs;
  shellHook = ''
    export LD_LIBRARY_PATH="${pkgs.gfortran.cc.lib}:${libPaths}:$LD_LIBRARY_PATH"
    export LIBRARY_PATH="${pkgs.gfortran.cc.lib}:${libPaths}"
    export CXX_PATH="${customStdenv.cc.cc.lib}"
    export LIB_PATHS="${libPaths}"
    export GIT_REV="$(git rev-parse HEAD)"
  '';
  shell = pkgs.mkShell.override { stdenv = customStdenv; } {
    name = "gargantext-shell";
    buildInputs = hsBuildInputs ++ nonhsBuildInputs;
    inherit shellHook;
  };
}
