{-# LANGUAGE TypeOperators #-}
{-# LANGUAGE DeriveGeneric #-}
{-# LANGUAGE QuasiQuotes   #-}

module Gargantext.MicroServices.ReverseProxy (
  microServicesProxyApp

  -- * Internals
  , replaceIt
  ) where

import Prelude

import Conduit
import Data.ByteString qualified as B
import Data.ByteString.Builder
import Data.ByteString.Char8 qualified as C8
import Data.Conduit.List qualified as CC
import Data.String
import Data.Text qualified as T
import Data.Text.Encoding qualified as TE
import GHC.Generics
import Gargantext.API.Admin.EnvTypes
import Gargantext.API.Admin.Settings.MicroServices
import Gargantext.API.Admin.Types
import Gargantext.Database.Prelude (hasConfig)
import Gargantext.Prelude
import Gargantext.Prelude.Config (gc_frame_write_url)
import Network.HTTP.ReverseProxy
import Network.HTTP.Types (hCacheControl, RequestHeaders, hReferer, ResponseHeaders)
import Network.HTTP.Types.Header (hHost)
import Network.URI as URI
import Network.Wai (Request, rawPathInfo, requestHeaders)
import Servant
import Servant.Auth.Swagger ()
import Servant.Server.Generic
import Text.RE.TDFA.ByteString
import Text.RE.Replace hiding (Capture)

--
-- Types
--

data ReverseProxyAPI mode = ReverseProxyAPI
  { notesProxy    :: mode :- "notes-proxy" :> NamedRoutes NotesProxy
  , notesSocketIo :: mode :- "socket.io"   :> NamedRoutes SocketIOProxy
  , meEndpoint    :: mode :- "me"          :> Raw
  } deriving Generic

data NotesProxy mode = NotesProxy
  { noteProxyEp :: mode :- Capture "frameId" T.Text
                        :> Raw
  } deriving Generic

data SocketIOProxy mode = SocketIOProxy
  { socketIoEp  :: mode :- QueryParam "noteId" T.Text :> Raw
  } deriving Generic

--
-- Server
--

microServicesProxyApp :: Env -> Application
microServicesProxyApp env = genericServe (server env)

server :: Env -> ReverseProxyAPI AsServer
server env = ReverseProxyAPI {
    notesProxy    = notesProxyImplementation env
  , notesSocketIo = socketIOProxyImplementation env
  , meEndpoint    = forwardServer env
  }

notesProxyImplementation :: Env -> NotesProxy AsServer
notesProxyImplementation env = NotesProxy {
    noteProxyEp = \_frameId -> forwardServer env
  }

socketIOProxyImplementation :: Env -> SocketIOProxy AsServer
socketIOProxyImplementation env = SocketIOProxy {
    socketIoEp = \_noteId -> forwardServer env
  }

forwardServer :: Env -> ServerT Raw m
forwardServer env =
  Tagged $ waiProxyToSettings forwardRequest proxySettings (env ^. env_manager)
  where

    microSrvSettings :: MicroServicesSettings
    microSrvSettings = env ^. env_settings . microservicesSettings

    pxyPort :: Int
    pxyPort = microSrvSettings ^. msProxyPort

    writeFrameURL :: T.Text
    writeFrameURL = env ^. hasConfig . gc_frame_write_url

    assetPath :: B.ByteString
    assetPath
      | snd forwardedHost /= 80
      = TE.encodeUtf8 $ writeFrameURL <> ":" <> T.pack (Prelude.show $ snd forwardedHost)
      | otherwise
      = TE.encodeUtf8 writeFrameURL

    removeProxyPath :: ByteString -> ByteString
    removeProxyPath = TE.encodeUtf8
                    . T.replace "/notes-proxy" ""
                    . TE.decodeUtf8

    proxySettings :: WaiProxySettings
    proxySettings = defaultWaiProxySettings { wpsProcessBody = \_req _res -> Just replaceRelativeLinks
                                            , wpsModifyResponseHeaders = \_req _res -> tweakResponseHeaders
                                            }

    replaceRelativeLinks :: ConduitT B.ByteString (Flush Builder) IO ()
    replaceRelativeLinks = CC.map (flushReplace assetPath)
      where

    noCache :: RequestHeaders -> RequestHeaders
    noCache hdrs = (hCacheControl, fromString "no-cache") :
      filter ((/=) hCacheControl . fst) hdrs

    newHost :: RequestHeaders -> RequestHeaders
    newHost hdrs =
      (hHost, fromString $ fst forwardedHost <> ":" <> Prelude.show (snd forwardedHost)) :
      filter ((/=) hHost . fst) hdrs

    newReferer :: RequestHeaders -> RequestHeaders
    newReferer hdrs =
      (hReferer, fromString $ "http://localhost:" <> Prelude.show pxyPort) :
      filter ((/=) hHost . fst) hdrs

    forwardedHost :: (String, Int)
    forwardedHost = fromMaybe (panicTrace "Invalid URI found in the proxied Request.") $ do
      URI{..} <- URI.parseURI (T.unpack writeFrameURL)
      uriAut  <- uriAuthority
      port    <- case uriPort uriAut of
        "" -> pure 80
        xs -> readMaybe (dropWhile (==':') xs)
      pure $ (uriRegName uriAut, port)

    mkProxyDest :: ProxyDest
    mkProxyDest =
      let (hst, prt) = forwardedHost
      in ProxyDest (C8.pack hst) prt

    -- | Forwards the request by substituting back the proxied address into the actual one.
    forwardRequest :: Request -> IO WaiProxyResponse
    forwardRequest originalRequest = do
      let proxiedReq = originalRequest {
                           rawPathInfo = removeProxyPath (rawPathInfo originalRequest)
                         , requestHeaders = (newReferer . newHost . noCache $ (requestHeaders originalRequest))
                         }
      pure $ WPRModifiedRequest proxiedReq mkProxyDest

tweakResponseHeaders :: ResponseHeaders -> ResponseHeaders
tweakResponseHeaders = Prelude.map tweakHeader
  where
    tweakHeader (k,v)
      | k == "Content-Security-Policy"
      = (k, fromString "default-src *; style-src * 'unsafe-inline'; script-src * 'unsafe-inline' 'unsafe-eval'; img-src * data: 'unsafe-inline'; connect-src * 'unsafe-inline'; frame-src *;")
      | otherwise
      = (k,v)

-- | Replaces the relative links in the proxied page content with proper urls.
flushReplace :: B.ByteString -> B.ByteString -> Flush Builder
flushReplace assetPath = Chunk . byteString . replaceIt assetPath

replaceIt :: B.ByteString -> B.ByteString -> B.ByteString
replaceIt assetPath htmlBlob =
  replaceAllCaptures ALL makeAbsolute $ htmlBlob *=~ [re|src="\/build\/|href="\/build\/|src="\/config|src="\/js\/|]
  where
    makeAbsolute _ _loc cap = case capturedText cap of
      "src=\"/build/"  -> Just $ "src=\"" <> assetPath <> "/build/"
      "href=\"/build/" -> Just $ "href=\"" <> assetPath <> "/build/"
      "src=\"/config"  -> Just $ "src=\"" <> assetPath <> "/config"
      "src=\"/js/"     -> Just $ "src=\"" <> assetPath <> "/js/"
      _                -> Just $ assetPath <> capturedText cap
