{-|
Module      : Gargantext.Core.Config.Worker
Description : Worker TOML file config
Copyright   : (c) CNRS, 2024
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

Although Async.Worker.Broker supports various Broker types, in
Gargantext we will only use PGMQ. This makes for easier config,
simpler design. Also, the DevOps stuff is simpler (providing multiple
brokers at the same time could lead to complexities in analyzing
what's going on).

If need arises, we could switch to a different broker by rewriting its
initialization. At the same time, sending and executing jobs should be
broker-agnostic.

-}


module Gargantext.Core.Config.Worker where

import Async.Worker.Broker.Types qualified as Broker
import Database.PostgreSQL.Simple qualified as PGS
import Gargantext.Core.Config.Types (unTOMLConnectInfo)
import Database.PGMQ.Types qualified as PGMQ
import Gargantext.Prelude
import Toml.Schema


type WorkerName = Text


data WorkerSettings =
  WorkerSettings {
      _wsDatabase     :: !PGS.ConnectInfo
      -- After this number of seconds, the job will be available again.
      
      -- You can set timeout for each job individually and this is the
      -- preferred method over using defaultVt.
    , _wsDefaultVisibilityTimeout :: PGMQ.VisibilityTimeout
    , _wsDefinitions  :: ![WorkerDefinition]
  } deriving (Show, Eq)
instance FromValue WorkerSettings where
  fromValue = parseTableFromValue $ do
    dbConfig <- reqKey "database"
    _wsDefinitions <- reqKey "definitions"
    let _wsDefaultVisibilityTimeout = 1
    return $ WorkerSettings { _wsDatabase = unTOMLConnectInfo dbConfig
                            , _wsDefinitions
                            , _wsDefaultVisibilityTimeout }

data WorkerDefinition =
  WorkerDefinition {
    _wdName   :: !WorkerName
  , _wdQueue  :: !Broker.Queue
  } deriving (Show, Eq)
instance FromValue WorkerDefinition where
  fromValue = parseTableFromValue $ do
    _wdName <- reqKey "name"
    queue <- reqKey "queue"
    return $ WorkerDefinition { _wdQueue = Broker.Queue queue, .. }

findDefinitionByName :: WorkerSettings -> WorkerName -> Maybe WorkerDefinition
findDefinitionByName (WorkerSettings { _wsDefinitions }) workerName =
  head $ filter (\wd -> _wdName wd == workerName) _wsDefinitions

-- wdToRedisBrokerInitParams :: WorkerDefinition -> Maybe BRedis.RedisBrokerInitParams
-- wdToRedisBrokerInitParams wd = BRedis.RedisBrokerInitParams <$> (wdToRedisConnectInfo wd)


