{-|
Module      : Gargantext.Core.Worker
Description : Asynchronous worker logic
Copyright   : (c) CNRS, 2024
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

-}

{-# LANGUAGE UndecidableInstances #-}
{-# OPTIONS_GHC -Wno-orphans      #-}  -- orphan HasNodeError IOException


module Gargantext.Core.Worker where


import Async.Worker.Broker.PGMQ (PGMQBroker)
import Async.Worker.Broker.Types (BrokerMessage, toA, getMessage)
import Async.Worker qualified as Worker
import Async.Worker.Types qualified as Worker
import Async.Worker.Types (HasWorkerBroker)
import Data.Text qualified as T
import Gargantext.API.Admin.Auth (forgotUserPassword)
import Gargantext.API.Admin.Auth.Types (ForgotPasswordAsyncParams(..))
import Gargantext.API.Node.Corpus.New (addToCorpusWithForm)
import Gargantext.API.Node.New (postNode')
import Gargantext.Core.Config (hasConfig)
import Gargantext.Core.Config.Worker (WorkerDefinition(..))
import Gargantext.Core.Worker.Broker (initBrokerWithDBCreate)
import Gargantext.Core.Worker.Env
import Gargantext.Core.Worker.Jobs.Types (Job(..))
import Gargantext.Database.Query.Table.User (getUsersWithEmail)
import Gargantext.Prelude
import Gargantext.Utils.Jobs.Monad ( MonadJobStatus(noJobHandle) )



-- | Spawn a worker with PGMQ broker
-- TODO:
--  - reduce size of DB pool
--  - progress report via notifications
--  - I think there is no point to save job result, as usually there is none (we have side-effects only)
--  - replace Servant.Job to use workers instead of garg API threads
withPGMQWorker :: (HasWorkerBroker PGMQBroker Job)
                => WorkerEnv
                -> WorkerDefinition
                -> (Async () -> Worker.State PGMQBroker Job -> IO ())
                -> IO ()
withPGMQWorker env (WorkerDefinition { .. }) cb = do
  let gargConfig = env ^. hasConfig
  broker <- initBrokerWithDBCreate gargConfig

  let state' = Worker.State { broker
                            , queueName = _wdQueue
                            , name = T.unpack _wdName
                            , performAction = performAction env
                            , onMessageReceived = Nothing
                            , onJobFinish = Nothing
                            , onJobTimeout = Nothing
                            , onJobError = Nothing }

  withAsync (Worker.run state') (\a -> cb a state')


performAction :: (HasWorkerBroker b Job)
              => WorkerEnv
              -> Worker.State b Job
              -> BrokerMessage b (Worker.Job Job)
              -> IO ()
performAction env _state bm = do
  let job' = toA $ getMessage bm
  case Worker.job job' of
    Ping -> putStrLn ("ping" :: Text)
    AddCorpusFormAsync { .. } -> runWorkerMonad env $ do
      liftBase $ putStrLn ("add corpus form" :: Text)
      addToCorpusWithForm _acf_user _acf_cid _acf_args (noJobHandle (Proxy :: Proxy WorkerMonad))
    ForgotPasswordAsync { _fpa_args = ForgotPasswordAsyncParams { email } } -> runWorkerMonad env $ do
      liftBase $ putStrLn ("forgot password: " <> email)
      us <- getUsersWithEmail (T.toLower email)
      case us of
        [u] -> forgotUserPassword u
        _ -> pure ()
    NewNodeAsync { .. } -> runWorkerMonad env $ do
      liftBase $ putStrLn ("new node async " :: Text)
      void $ postNode' _nna_authenticatedUser _nna_node_id _nna_postNode
      return ()
    GargJob { _gj_garg_job } -> putStrLn ("Garg job: " <> show _gj_garg_job :: Text)
