{-|
Module      : Gargantext.API.Admin.Auth.Types
Description : Types for Server API Auth Module
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX
-}

{-# LANGUAGE TemplateHaskell    #-}
{-# LANGUAGE DerivingStrategies #-}
{-# LANGUAGE DeriveAnyClass     #-}

module Gargantext.API.Admin.Auth.Types
  ( -- * Types
    AuthRequest(..)
  , AuthResponse(..)
  , Token
  , TreeId
  , CheckAuth(..)
  , AuthenticatedUser(..)
  , AuthContext
  , AuthenticationError(..)
  , PathId(..)
  , Email
  , Password
  , ForgotPasswordRequest(..)
  , ForgotPasswordResponse(..)
  , ForgotPasswordAsyncParams(..)
  , ForgotPasswordGet(..)
  , RemoteTransferPublicKey(..)

  -- * Lenses
  , auth_node_id
  , auth_user_id
  , authRes_token
  , authRes_tree_id
  , authRes_user_id
  , authRes_remote_transfer_pub_key

  -- * Combinators
  , pubKeyToRemotePubKey
  , remotePubKeyToPubKey
  ) where

import Crypto.JWT qualified as Jose
import Crypto.PubKey.RSA qualified as RSA
import Data.Aeson.TH qualified as JSON
import Data.Aeson.Types (genericParseJSON, defaultOptions, genericToJSON)
import Data.ASN1.BinaryEncoding
import Data.ASN1.Encoding qualified as ASN1
import Data.ASN1.Types (toASN1, fromASN1)
import Data.ByteString.Base64 qualified as Base64
import Data.List (tail)
import Data.Swagger ( ToSchema(..), genericDeclareNamedSchema )
import Data.Text.Encoding qualified as TE
import Data.Text qualified as T
import Data.X509 qualified as X509
import Gargantext.Core.Types.Individu (Username, GargPassword(..), arbitraryUsername, arbitraryPassword)
import Gargantext.Core.Utils.Prefix (unPrefix, unPrefixSwagger)
import Gargantext.Database.Admin.Types.Node (NodeId(..), ListId, DocId, UserId (..))
import Gargantext.Prelude hiding (reverse)
import Prelude (String)
import Servant.Auth.Server
import Test.QuickCheck.Arbitrary (Arbitrary, arbitrary)
import Test.QuickCheck (elements)

---------------------------------------------------

-- | Main types for AUTH API
data AuthRequest = AuthRequest { _authReq_username :: Username
                               , _authReq_password :: GargPassword
                               }
  deriving (Generic)

newtype RemoteTransferPublicKey =
  RemoteTransferPublicKey { _RemoteTransferPublicKey :: T.Text }
  deriving stock (Generic, Eq, Show)
  deriving newtype (ToJSON, FromJSON)
  deriving anyclass (ToSchema)

instance NFData RemoteTransferPublicKey where

pubKeyToRemotePubKey :: RSA.PublicKey -> RemoteTransferPublicKey
pubKeyToRemotePubKey pubKey =
  let x509pubKey = X509.PubKeyRSA pubKey
      ans1Enc    = ASN1.encodeASN1' DER ((toASN1 x509pubKey) [])
      in RemoteTransferPublicKey $ TE.decodeUtf8 (Base64.encode ans1Enc)

remotePubKeyToPubKey :: RemoteTransferPublicKey -> Either String RSA.PublicKey
remotePubKeyToPubKey (RemoteTransferPublicKey pkeyTxt) = do
  unwrappedB64 <- Base64.decode (TE.encodeUtf8 pkeyTxt)
  case ASN1.decodeASN1' DER unwrappedB64 of
    Left asn1Err  -> Left $ show asn1Err
    Right asn1Obj -> do
      (x509Ty, _) <- fromASN1 asn1Obj
      case x509Ty of
        X509.PubKeyRSA pk -> pure (pk { RSA.public_size = 256 })
        _                 -> Left "remotePubKeyToPubKey: x509 incompatible type found."

data AuthResponse = AuthResponse { _authRes_token   :: Token
                                 , _authRes_tree_id :: TreeId
                                 , _authRes_user_id :: UserId
                                 -- | The remote transfer public key which the
                                 -- browser can save and later use in transfer
                                 -- requests.
                                 , _authRes_remote_transfer_pub_key :: RemoteTransferPublicKey
                                 }
  deriving (Generic, Eq, Show)

instance NFData AuthResponse where

type Token  = Text
type TreeId = NodeId

data CheckAuth = InvalidUser | InvalidPassword | Valid Token TreeId UserId
  deriving (Eq)

data AuthenticatedUser = AuthenticatedUser
  { _auth_node_id :: NodeId
  , _auth_user_id :: UserId
  } deriving (Generic, Show, Eq)

makeLenses ''AuthenticatedUser

instance ToSchema AuthenticatedUser where
  declareNamedSchema = genericDeclareNamedSchema (unPrefixSwagger "_authUser_")

data AuthenticationError
  = LoginFailed NodeId UserId Jose.Error
  | InvalidUsernameOrPassword
  | UserNotAuthorized UserId Text
  deriving (Show, Eq)

-- TODO-SECURITY why is the CookieSettings necessary?
type AuthContext = '[JWTSettings, CookieSettings] -- , BasicAuthCfg

-- | Instances
instance ToSchema AuthRequest where
  declareNamedSchema = genericDeclareNamedSchema (unPrefixSwagger "_authReq_")

instance Arbitrary AuthRequest where
  arbitrary = elements [ AuthRequest u p
                       | u <- arbitraryUsername
                       , p <- arbitraryPassword
                       ]

instance ToSchema AuthResponse where
  declareNamedSchema = genericDeclareNamedSchema (unPrefixSwagger "_authRes_")
instance Arbitrary AuthResponse where
  arbitrary = elements [ AuthResponse to' tr u k
                       | to' <- ["token0", "token1"]
                       , tr <- map UnsafeMkNodeId [1..3]
                       , u <-  map UnsafeMkUserId [1..3]
                       , k <-  pure $ RemoteTransferPublicKey "dummy-pubkey"
                       ]

data PathId = PathNode NodeId | PathNodeNode ListId DocId


---------------------------

type Email = Text
type Password = Text

data ForgotPasswordRequest = ForgotPasswordRequest { _fpReq_email :: Email }
  deriving (Generic)
instance ToSchema ForgotPasswordRequest where
  declareNamedSchema = genericDeclareNamedSchema (unPrefixSwagger "_fpReq_")

data ForgotPasswordResponse = ForgotPasswordResponse { _fpRes_status :: Text }
  deriving (Generic)
instance ToSchema ForgotPasswordResponse where
  declareNamedSchema = genericDeclareNamedSchema (unPrefixSwagger "_fpRes_")

data ForgotPasswordGet = ForgotPasswordGet {_fpGet_password :: Password}
  deriving (Generic)
instance ToSchema ForgotPasswordGet where
  declareNamedSchema = genericDeclareNamedSchema (unPrefixSwagger "_fpGet_")

newtype ForgotPasswordAsyncParams =
  ForgotPasswordAsyncParams { email :: Text }
  deriving (Generic, Show, Eq)
instance FromJSON ForgotPasswordAsyncParams where
  parseJSON = genericParseJSON defaultOptions
instance ToJSON ForgotPasswordAsyncParams where
  toJSON = genericToJSON defaultOptions
instance ToSchema ForgotPasswordAsyncParams

--
-- Lenses
--

makeLenses ''AuthResponse

--
-- JSON instances
--

$(deriveJSON (JSON.defaultOptions { JSON.fieldLabelModifier = tail . dropWhile ((/=) '_') . tail }) ''AuthenticatedUser)
$(deriveJSON (unPrefix "_authReq_") ''AuthRequest)
$(deriveJSON (unPrefix "_authRes_") ''AuthResponse)
$(deriveJSON (unPrefix "_fpReq_") ''ForgotPasswordRequest)
$(deriveJSON (unPrefix "_fpRes_") ''ForgotPasswordResponse)
$(deriveJSON (unPrefix "_fpGet_") ''ForgotPasswordGet)

--
-- JWT instances
--

instance ToJWT AuthenticatedUser
instance FromJWT AuthenticatedUser
