{-|
Module      : Gargantext.API.Node.Corpus.Types
Description :
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX
-}


module Gargantext.API.Node.Corpus.Types where

import Control.Lens ( (?~) )
import Control.Monad.Fail (fail)
import Data.Aeson ( Value(..), (.:), (.:?), (.=), withText, object, withObject )
import Data.Aeson.Types (unexpected)
import Data.Swagger
import Data.Text qualified as T
import Gargantext.API.Admin.Orchestrator.Types qualified as Types
import Gargantext.Database.Action.Flow.Types (DataOrigin(..))
import Gargantext.Prelude
import PUBMED.Types qualified as PUBMED

type EPOAPIToken = Text
type EPOAPIUser = Text

data Database = Empty
              | OpenAlex
              | PubMed (Maybe PUBMED.APIKey)
              | Arxiv
              | HAL
              | IsTex
              | Isidore
              | EPO (Maybe EPOAPIUser) (Maybe EPOAPIToken)
  deriving (Eq, Show, Generic)

instance FromJSON Database where
  parseJSON = withObject "Database" $ \o -> do
    db <- o .: "db"
    case db of
      "Empty" -> pure Empty
      "OpenAlex" -> pure OpenAlex
      "PubMed" -> do
        mAPIKey <- o .:? "api_key"
        pure $ PubMed mAPIKey
      "Arxiv" -> pure Arxiv
      "HAL" -> pure HAL
      "IsTex" -> pure IsTex
      "Isidore" -> pure Isidore
      "EPO" -> do
        mAPIUser <- o .:? "api_user"
        mAPIToken <- o .:? "api_token"
        pure $ EPO mAPIUser mAPIToken
      s -> unexpected (String s)
instance ToJSON Database where
  toJSON (PubMed mAPIKey) = object [ "db" .= toJSON ("PubMed" :: Text)
                                   , "api_key" .= toJSON mAPIKey ]
  toJSON (EPO mAPIUser mAPIToken) = object [ "db" .= toJSON ("EPO" :: Text)
                                           , "api_user" .= toJSON mAPIUser
                                           , "api_token" .= toJSON mAPIToken ]
  toJSON t = object [ "db" .= toJSON (show t :: Text) ]
instance ToSchema Database where
  declareNamedSchema = genericDeclareNamedSchemaUnrestricted defaultSchemaOptions

datafield2origin :: Datafield -> DataOrigin
datafield2origin (External Empty)                    = InternalOrigin Types.IsTex
datafield2origin (External OpenAlex)                 = ExternalOrigin Types.OpenAlex
datafield2origin (External (PubMed mAPIKey))         = ExternalOrigin (Types.PubMed mAPIKey)
datafield2origin (External Arxiv)                    = ExternalOrigin Types.Arxiv
datafield2origin (External HAL)                      = ExternalOrigin Types.HAL
datafield2origin (External IsTex)                    = ExternalOrigin Types.IsTex
datafield2origin (External Isidore)                  = ExternalOrigin Types.Isidore
datafield2origin (External (EPO mAPIUser mAPIToken)) = ExternalOrigin (Types.EPO mAPIUser mAPIToken)
-- | This isn't really used
datafield2origin _                                   = InternalOrigin Types.IsTex

------------------------------------------------------------------------
data Datafield = Gargantext
               | External Database
               | Web
               | Files
  deriving (Eq, Show, Generic)

instance FromJSON Datafield where
  parseJSON (String "Gargantext") = pure Gargantext
  parseJSON (String "Web") = pure Web
  parseJSON (String "Files") = pure Files
  parseJSON (Object o) = do
    db <- o .: "External"
    pure $ External db
  parseJSON x = withText "Datafield" (\text ->
    fail $ "Cannot match pattern '<db>' for string " <> T.unpack text) x

instance ToJSON Datafield where
  toJSON (External db) = toJSON $ object [ ("External", toJSON db) ]
  toJSON s = toJSON (show s :: Text)

instance ToSchema Datafield where
  declareNamedSchema _ = do
    pure $ NamedSchema (Just "Datafield") $ mempty
      & type_ ?~ SwaggerObject
