{-|
Module      : Gargantext.API.Node.Corpus.Export.Types
Description : Types for Gargantext.API.Node.Corpus.Export
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX
-}

{-# LANGUAGE TemplateHaskell #-}
{-# LANGUAGE TypeOperators   #-}

module Gargantext.API.Node.Corpus.Export.Types where

import Data.ByteString.Lazy qualified as BSL
import Data.Swagger ( ToSchema(..), genericDeclareNamedSchema, ToParamSchema(..), NamedSchema(..), binarySchema )
import Data.Time.Clock (UTCTime)
import Data.Version (Version)
import Gargantext.API.Node.Document.Export.Types qualified as DocumentExport
import Gargantext.API.Ngrams.Types (NgramsTerm)
import Gargantext.Core.Types ( CorpusId, ListId, TODO )
import Gargantext.Core.Utils.Prefix (unPrefix, unPrefixSwagger)
import Gargantext.Database.Admin.Types.Hyperdata.Corpus (HyperdataCorpus)
import Gargantext.Database.Admin.Types.Hyperdata.Document (HyperdataDocument)
import Gargantext.Database.Admin.Types.Hyperdata.List (HyperdataList)
import Gargantext.Database.Admin.Types.Node (ContextId, NodeId)
import Gargantext.Prelude
import Gargantext.Prelude.Crypto.Hash qualified as H
import Servant (Accept(..), MimeRender(mimeRender), MimeUnrender(mimeUnrender), OctetStream)


-- Corpus Export
data Corpus =
  Corpus { _c_corpus :: [DocumentExport.Document]
         , _c_hash   :: Hash
         } deriving (Generic)

type Hash = Text
-------
instance ToSchema Corpus where
  declareNamedSchema = genericDeclareNamedSchema (unPrefixSwagger "_c_")
-------
instance ToParamSchema Corpus where
  toParamSchema _ = toParamSchema (Proxy :: Proxy TODO)
$(deriveJSON (unPrefix "_c_") ''Corpus)




-- | Wrapper around 'ByteString' to return an SQLite db containing
-- corpus
newtype CorpusSQLite =
  CorpusSQLite { _cs_bs :: BSL.ByteString }
  deriving (Generic, NFData)
instance Accept CorpusSQLite where
  contentType _ = contentType (Proxy :: Proxy OctetStream)
instance MimeRender OctetStream CorpusSQLite where
  mimeRender _ (CorpusSQLite bs) = bs
-- | Needed for tests
instance MimeUnrender OctetStream CorpusSQLite where
  mimeUnrender _ bs = Right $ CorpusSQLite { _cs_bs = bs }
instance ToSchema CorpusSQLite where
  declareNamedSchema _ = pure $ NamedSchema (Just "CorpusSQLite") binarySchema

                         
-- | Contents of the SQLite export DB
-- (having such datatype makes it easier to coherently implement import/export)
data CorpusSQLiteData =
  CorpusSQLiteData { _csd_version                  :: Version
                   , _csd_cId                      :: CorpusId
                   , _csd_lId                      :: ListId
                   , _csd_created                  :: UTCTime
                                                  
                   , _csd_corpus_name              :: Text
                   , _csd_corpus_hash              :: Maybe H.Hash
                   , _csd_corpus_parent            :: Maybe NodeId
                   , _csd_corpus_hyperdata         :: HyperdataCorpus
                                                  
                   , _csd_list_name                :: Text
                   , _csd_list_parent              :: Maybe NodeId
                   , _csd_list_hyperdata           :: HyperdataList
                                                  
                   , _csd_contexts                 :: [(NodeId, Text, UTCTime, HyperdataDocument)]

                   , _csd_map_context_ngrams       :: Map ContextId (Set NgramsTerm)
                   , _csd_stop_context_ngrams      :: Map ContextId (Set NgramsTerm)
                   , _csd_candidate_context_ngrams :: Map ContextId (Set NgramsTerm)
                   } deriving (Show, Eq, Generic)

