{--| Prelude module for our API specs, with utility functions to get us started quickly. -}

module Test.API.Prelude
  ( newCorpusForUser
  , newPrivateFolderForUser
  , newPublicFolderForUser
  , newFolderForUser
  , getRootPublicFolderIdForUser
  , getRootPrivateFolderIdForUser
  , myUserNodeId
  , checkEither
  , shouldFailWith
  ) where

import Data.Aeson qualified as JSON
import Data.Text qualified as T
import Gargantext.API.Errors
import Gargantext.Core.Types.Individu
import Gargantext.Core.Types (NodeId, NodeType(..))
import Gargantext.Core.Worker.Env ()  -- instance HasNodeError
import Gargantext.Database.Action.User
import Gargantext.Database.Admin.Types.Hyperdata.Corpus
import Gargantext.Database.Query.Table.Node
import Gargantext.Database.Query.Table.Node.User (getUserByName)
import Gargantext.Database.Query.Tree.Root
import Gargantext.Database.Schema.Node (_node_id)
import Gargantext.Prelude hiding (get)
import Prelude (fail)
import Servant.Client.Core
import Test.Database.Types
import Test.Tasty.HUnit (Assertion, (@?=))

checkEither :: (Show a, Monad m) => m (Either a b) -> m b
checkEither = fmap (either (\x -> panicTrace $ "checkEither:" <> T.pack (show x)) identity)

newCorpusForUser :: TestEnv -> T.Text -> IO NodeId
newCorpusForUser env uname = flip runReaderT env $ runTestMonad $ do
  uid      <- getUserId (UserName uname)
  parentId <- getRootId (UserName uname)
  let corpusName = "Test_Corpus"
  (corpusId:_) <- mk (Just corpusName) (Nothing :: Maybe HyperdataCorpus) parentId uid
  pure corpusId

newFolderForUser :: TestEnv -> T.Text -> T.Text -> IO NodeId
newFolderForUser env uname folderName = flip runReaderT env $ runTestMonad $ do
  uid      <- getUserId (UserName uname)
  parentId <- getRootId (UserName uname)
  insertNode NodeFolder (Just folderName) Nothing parentId uid

-- | Generate a 'Node' where we can append more data into, a bit reminiscent to the
-- \"Private\" root node we use in the real Gargantext.
newPrivateFolderForUser :: TestEnv -> T.Text -> IO NodeId
newPrivateFolderForUser env uname = flip runReaderT env $ runTestMonad $ do
  uid      <- getUserId (UserName uname)
  parentId <- getRootId (UserName uname)
  let nodeName = "NodeFolderPrivate"
  insertNode NodeFolderPrivate (Just nodeName) Nothing parentId uid

newPublicFolderForUser :: TestEnv -> T.Text -> IO NodeId
newPublicFolderForUser env uname = flip runReaderT env $ runTestMonad $ do
  uid      <- getUserId (UserName uname)
  parentId <- getRootId (UserName uname)
  let nodeName = "NodeFolderPublic"
  insertNode NodeFolderPublic (Just nodeName) Nothing parentId uid

getRootPublicFolderIdForUser :: TestEnv -> User -> IO NodeId
getRootPublicFolderIdForUser env uname = flip runReaderT env $ runTestMonad $ do
  _node_id <$> (getUserId uname >>= getUserRootPublicNode)

getRootPrivateFolderIdForUser :: TestEnv -> User -> IO NodeId
getRootPrivateFolderIdForUser env uname = flip runReaderT env $ runTestMonad $ do
  _node_id <$> (getUserId uname >>= getUserRootPrivateNode)

myUserNodeId :: TestEnv -> T.Text -> IO NodeId
myUserNodeId env uname = flip runReaderT env $ runTestMonad $ do
  _node_id <$> getUserByName uname

shouldFailWith :: Show a => Either ClientError a -> BackendErrorCode -> Assertion
action `shouldFailWith` backendError = case action of
  Right{} -> fail "Expected action to fail, but it didn't."
  Left fr@(FailureResponse _req res)
    | Right FrontendError{..} <- JSON.eitherDecode (responseBody res)
    -> fe_type @?= backendError
    | otherwise
    -> fail $ "FailureResponse didn't have FrontendError: " <> show fr
  _xs -> fail $ "Unexpected ClientError: " <> show _xs
