{-|
Module      : CLI.Types
Description : CLI types
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

-}


{-# LANGUAGE DuplicateRecordFields #-}

module CLI.Types where

import Data.String
import Data.Text (Text)
import Gargantext.API.Admin.EnvTypes (Mode)
import Gargantext.Core.Config.Types (SettingsFile(..))
import Gargantext.Core.Types (NodeId)
import Prelude

newtype CorpusFile = CorpusFile { _CorpusFile :: FilePath }
  deriving (Show, Eq, IsString)

newtype TermListFile = TermListFile { _TermsListFile :: FilePath }
  deriving (Show, Eq, IsString)

newtype OutputFile = OutputFile { _OutputFile :: FilePath }
  deriving (Show, Eq, IsString)

data ObfuscateDBArgs = ObfuscateDBArgs {
    dbHost     :: !Text
  , dbPort     :: !Int
  , dbName     :: !Text
  , dbUser     :: !Text
  , dbPassword :: !Text
  } deriving (Show, Eq)

data AdminArgs = AdminArgs
  { settingsPath :: !SettingsFile
  , emails       :: [String]
  } deriving (Show, Eq)

data ImportFunction
  = IF_corpus
  | IF_corpusTsvHal
  | IF_annuaire
  deriving (Show, Eq, Enum, Bounded)

data ImportArgs = ImportArgs
  { imp_function :: !ImportFunction
  , imp_user     :: !Text
  , imp_name     :: !Text
  , imp_settings :: !SettingsFile
  , imp_corpus_path :: !FilePath
  } deriving (Show, Eq)

data IniArgs = IniArgs
  { ini_path  :: !(Maybe FilePath)
  , toml_path :: !(Maybe FilePath)
  , dry_run   :: !Bool
  } deriving (Show, Eq)

data InitArgs = InitArgs
  { init_settings :: !SettingsFile
  } deriving (Show, Eq)

data InvitationsArgs = InvitationsArgs
  { inv_settings :: !SettingsFile
  , inv_user    :: !Text
  , inv_node_id :: !NodeId
  , inv_email   :: !Text
  } deriving (Show, Eq)

data PhyloArgs = PhyloArgs
  { phylo_config :: !FilePath
  } deriving (Show, Eq)

data UpgradeArgs = UpgradeArgs
  { upgrade_settings :: !SettingsFile
  } deriving (Show, Eq)

data GoldenFileDiffArgs = GoldenFileDiffArgs
  { gdf_expected :: !FilePath
  , gdf_actual   :: !FilePath
  } deriving (Show, Eq)

data CLIRoutes
  = CLIR_list
  | CLIR_export FilePath
  deriving (Show, Eq)

data CLIServer
  = CLIS_start ServerArgs
  | CLIS_startAll ServerArgs
  | CLIS_version
  deriving (Show, Eq)

data ServerArgs = ServerArgs
  { server_mode :: !Mode
  , server_toml :: !SettingsFile }
  deriving (Show, Eq)

data CLIWorker
  = CLIW_run WorkerArgs
  | CLIW_runSingleJob WorkerSingleArgs
  | CLIW_runAll WorkerAllArgs
  | CLIW_stats WorkerStatsArgs
  deriving (Show, Eq)

data WorkerArgs = WorkerArgs
  { worker_toml       :: !SettingsFile
  , worker_name       :: !Text
  , worker_count      :: !Int
  } deriving (Show, Eq)

data WorkerSingleArgs = WorkerSingleArgs
  { worker_toml       :: !SettingsFile
  , worker_name       :: !Text
  } deriving (Show, Eq)

data WorkerAllArgs = WorkerAllArgs
  { worker_toml       :: !SettingsFile
  } deriving (Show, Eq)

data WorkerStatsArgs = WorkerStatsArgs
  { ws_toml       :: !SettingsFile
  } deriving (Show, Eq)

data CLICmd
  = CCMD_admin !AdminArgs
  | CCMD_clean_csv_corpus
  | CCMD_filter_terms_and_cooc !CorpusFile !TermListFile !OutputFile
  | CCMD_golden_file_diff !GoldenFileDiffArgs
  | CCMD_import !ImportArgs
  | CCMD_ini !IniArgs
  | CCMD_init !InitArgs
  | CCMD_invitations !InvitationsArgs
  | CCMD_obfuscate_db !ObfuscateDBArgs
  | CCMD_phylo !PhyloArgs
  | CCMD_phylo_profile
  | CCMD_routes !CLIRoutes
  | CCMD_server !CLIServer
  | CCMD_upgrade !UpgradeArgs
  | CCMD_worker !CLIWorker
  deriving (Show, Eq)

data CLI =
  CLISub CLICmd
  deriving (Show, Eq)


