{-|
Module      : Test.Database.Operations.PublishNode
Description : GarganText database tests
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX
-}

{-# LANGUAGE QuasiQuotes   #-}
{-# LANGUAGE TupleSections #-}

module Test.Database.Operations.PublishNode where

import Prelude

import Control.Monad.Reader
import Gargantext.Core
import Gargantext.Core.Types.Individu
import Gargantext.Database.Action.User (getUserId)
import Gargantext.Database.Admin.Types.Node
import Gargantext.Database.Prelude (DBCmd)
import Gargantext.Database.Query.Table.Node
import Gargantext.Database.Query.Table.NodeNode
import Gargantext.Database.Schema.Node (NodePoly(..))
import Test.API.Prelude (newPrivateFolderForUser, newPublicFolderForUser, alice)
import Test.Database.Types
import Test.Tasty.HUnit

publishStrict :: SourceId -> TargetId -> DBCmd err ()
publishStrict = publishNode NPP_publish_no_edits_allowed

publishLenient :: SourceId -> TargetId -> DBCmd err ()
publishLenient = publishNode NPP_publish_edits_only_owner_or_super

testGetUserRootPublicNode :: TestEnv -> Assertion
testGetUserRootPublicNode testEnv = do
  alicePublicFolder <- flip runReaderT testEnv $ runTestMonad $ do
    aliceId <- getUserId (UserName "alice")
    getUserRootPublicNode aliceId
  _node_typename alicePublicFolder @?= (toDBid NodeFolderPublic)

testIsReadOnlyWorks :: TestEnv -> Assertion
testIsReadOnlyWorks testEnv = do
  alicePrivateFolderId <- newPrivateFolderForUser testEnv alice
  alicePublicFolderId  <- newPublicFolderForUser testEnv alice
  flip runReaderT testEnv $ runTestMonad $ do
    -- Create a corpus, by default is not read only
    aliceUserId <- getUserId (UserName "alice")
    corpusId    <- insertDefaultNode NodeCorpus alicePrivateFolderId aliceUserId

    isNodeReadOnly corpusId >>= liftIO . (@?= False)

    -- Publish the node, then check that's now public.
    publishStrict (SourceId corpusId) (TargetId alicePublicFolderId)
    isNodeReadOnly corpusId >>= liftIO . (@?= True)

    -- Finally check that if we unpublish, the node is back to normal
    unpublishNode (SourceId corpusId) (TargetId alicePublicFolderId)
    isNodeReadOnly corpusId >>= liftIO . (@?= False)

-- | In this test, we check that if we publish the root of a subtree,
-- then all the children (up to the first level) are also marked read-only.
testPublishRecursiveFirstLevel :: TestEnv -> Assertion
testPublishRecursiveFirstLevel testEnv = do
  alicePrivateFolderId <- newPrivateFolderForUser testEnv alice
  alicePublicFolderId  <- newPublicFolderForUser testEnv alice
  flip runReaderT testEnv $ runTestMonad $ do
    -- Create a corpus, by default is not read only
    aliceUserId    <- getUserId (UserName "alice")
    aliceFolderId  <- insertDefaultNode NodeFolder alicePrivateFolderId aliceUserId
    corpusId       <- insertDefaultNode NodeCorpus aliceFolderId aliceUserId

    publishStrict (SourceId aliceFolderId) (TargetId alicePublicFolderId)

    isNodeReadOnly aliceFolderId >>= liftIO . (@?= True)
    isNodeReadOnly corpusId      >>= liftIO . (@?= True)

-- | In this test, we check that if we publish the root of a subtree,
-- then all the children of the children are also marked read-only.
testPublishRecursiveNLevel :: TestEnv -> Assertion
testPublishRecursiveNLevel testEnv = do
  alicePrivateFolderId <- newPrivateFolderForUser testEnv alice
  alicePublicFolderId  <- newPublicFolderForUser testEnv alice
  flip runReaderT testEnv $ runTestMonad $ do
    -- Create a corpus, by default is not read only
    aliceUserId    <- getUserId (UserName "alice")
    aliceFolderId  <- insertDefaultNode NodeFolder alicePrivateFolderId aliceUserId
    aliceSubFolderId <- insertDefaultNode NodeFolder aliceFolderId aliceUserId
    corpusId       <- insertDefaultNode NodeCorpus aliceSubFolderId aliceUserId

    publishStrict (SourceId aliceFolderId) (TargetId alicePublicFolderId)

    isNodeReadOnly aliceFolderId    >>= liftIO . (@?= True)
    isNodeReadOnly aliceSubFolderId >>= liftIO . (@?= True)
    isNodeReadOnly corpusId         >>= liftIO . (@?= True)

testPublishLenientWorks :: TestEnv -> Assertion
testPublishLenientWorks testEnv = do
  alicePrivateFolderId <- newPrivateFolderForUser testEnv alice
  alicePublicFolderId  <- newPublicFolderForUser testEnv alice
  flip runReaderT testEnv $ runTestMonad $ do
    aliceUserId <- getUserId (UserName "alice")
    corpusId    <- insertDefaultNode NodeCorpus alicePrivateFolderId aliceUserId
    publishLenient (SourceId corpusId) (TargetId alicePublicFolderId)
    isNodeReadOnly corpusId >>= liftIO . (@?= True)
