module Test.Integration.PGMQ.Simple
  ( pgmqSimpleTests )
where

import Control.Exception (bracket)
import Data.Maybe (isJust)
import Database.PostgreSQL.Simple qualified as PSQL
import Database.PGMQ.Simple qualified as PGMQ
import Database.PGMQ.Types qualified as PGMQ
import Test.Hspec
import Test.Integration.Utils (getPSQLEnvConnectInfo, randomQueueName)


data TestEnv =
  TestEnv {
    conn  :: PSQL.Connection
  , queue :: PGMQ.Queue
  }
    
-- NOTE These tests expect a local pgmq server runnign on port 5432.

testQueuePrefix :: PGMQ.Queue
testQueuePrefix = "test_pgmq"

setUpConn :: IO TestEnv
setUpConn = do
  connInfo <- getPSQLEnvConnectInfo
  conn <- PSQL.connect connInfo
  queue <- randomQueueName testQueuePrefix
  return $ TestEnv { conn, queue }

dropConn :: TestEnv -> IO ()
dropConn (TestEnv { conn }) = do
  PSQL.close conn

withConn :: (TestEnv -> IO ()) -> IO ()
withConn = bracket setUpConn dropConn

withPGMQ :: (TestEnv -> IO ()) -> IO ()
withPGMQ f = withConn $ \testEnv -> bracket (setUpPGMQ testEnv) (tearDownPGMQ testEnv) (\_ -> f testEnv)
  where
    setUpPGMQ (TestEnv { conn, queue }) = do
      PGMQ.initialize conn
      PGMQ.dropQueue conn queue
      PGMQ.createQueue conn queue

    tearDownPGMQ (TestEnv { conn, queue }) _ = do
      PGMQ.dropQueue conn queue

      
pgmqSimpleTests :: Spec
pgmqSimpleTests = parallel $ around withPGMQ $ describe "PGMQ Simple" $ do
  -- it "can get metrics for non-existing queue" $ \(TestEnv { conn, queue }) -> do
  --   -- first of all, this should also work for non-existing queues
  --   metrics <- PGMQ.getMetrics conn queue
  --   metrics `shouldSatisfy` isNothing

  it "can get metrics for an empty queue" $ \(TestEnv { conn, queue }) -> do
    metrics <- PGMQ.getMetrics conn queue
    metrics `shouldSatisfy` isJust
    (PGMQ.queueLength <$> metrics) `shouldBe` (Just 0)

  it "listQueues properly returns our queue" $ \(TestEnv { conn, queue }) -> do
    queues <- PGMQ.listQueues conn
    ((\(PGMQ.QueueInfo { queueName }) -> queueName) <$> queues) `shouldContain` [queue]
    
