{-|
  Generic Broker tests. All brokers should satisfy them.
-}

{-# LANGUAGE AllowAmbiguousTypes #-}
{-# LANGUAGE FlexibleContexts #-}
{-# LANGUAGE ScopedTypeVariables #-}

module Test.Integration.Broker
 ( brokerTests
 , pgmqBrokerInitParams )
where

import Async.Worker.Broker.PGMQ (BrokerInitParams(..), PGMQBroker)
import Async.Worker.Broker.Types qualified as BT
import Control.Exception (bracket)
import Data.Aeson (ToJSON(..), FromJSON(..), withText)
import Data.Text qualified as T
import Test.Hspec
import Test.Integration.PGMQ.Simple (getEnvConnectInfo)


data TestEnv b =
  TestEnv { broker :: BT.Broker b Message
          , queue  :: BT.Queue }

testQueue :: BT.Queue
testQueue = "test"


data Message =
  Message { text :: String }
  deriving (Show, Eq)
instance ToJSON Message where
  toJSON (Message { text }) = toJSON text
instance FromJSON Message where
  parseJSON = withText "Message" $ \text -> do
    pure $ Message { text = T.unpack text }


pgmqBrokerInitParams :: IO (BT.BrokerInitParams PGMQBroker Message)
pgmqBrokerInitParams = do
  connectInfo <- getEnvConnectInfo
  pure $ PGMQBrokerInitParams connectInfo


withBroker :: (BT.HasBroker b Message)
           => BT.BrokerInitParams b Message
           -> (TestEnv b -> IO ())
           -> IO ()
withBroker bInitParams = bracket (setUpBroker bInitParams) tearDownBroker
  where
    -- NOTE I need to pass 'b' again, otherwise GHC can't infer the
    -- type of 'b' (even with 'ScopedTypeVariables' turned on)
    setUpBroker :: (BT.HasBroker b Message)
                => BT.BrokerInitParams b Message -> IO (TestEnv b)
    setUpBroker bInit = do
      b <- BT.initBroker bInit
      
      BT.dropQueue b testQueue
      BT.createQueue b testQueue
      
      return $ TestEnv { broker = b
                       , queue  = testQueue }

    tearDownBroker (TestEnv { broker, queue }) = do
      BT.dropQueue broker queue
      BT.deinitBroker broker


brokerTests :: (BT.HasBroker b Message)
            => BT.BrokerInitParams b Message -> Spec
brokerTests bInitParams =
  sequential $ around (withBroker bInitParams) $ describe "Broker tests" $ do
    it "can send and receive a message" $ \(TestEnv { broker, queue }) -> do
      BT.dropQueue broker queue
      BT.createQueue broker queue
      let msg = Message { text = "test" }
      BT.sendMessage broker queue (BT.toMessage msg)
      msg2 <- BT.readMessageWaiting broker queue
      putStrLn $ "[messageId] " <> show (BT.messageId msg2)
      msg `shouldBe` (BT.toA $ BT.getMessage msg2)

