module Test.Integration.PGMQ.Simple
  ( pgmqSimpleTests
  , getEnvConnectInfo )
where

import Async.Worker.Broker qualified as B
import Control.Exception (bracket)
import Data.Maybe (isJust, fromMaybe)
import Database.PostgreSQL.Simple qualified as PSQL
import Database.PGMQ.Simple qualified as PGMQ
import Database.PGMQ.Types qualified as PGMQ
import System.Environment (lookupEnv)
import Test.Hspec


data TestEnv =
  TestEnv {
    conn  :: PSQL.Connection
  , queue :: B.Queue
  }
    
-- NOTE These tests expect a local pgmq server runnign on port 5432.

testQueue :: B.Queue
testQueue = "test"

getEnvConnectInfo :: IO PSQL.ConnectInfo
getEnvConnectInfo = do
  pgUser <- lookupEnv "POSTGRES_USER"
  pgDb <- lookupEnv "POSTGRES_DB"
  pgPass <- lookupEnv "POSTGRES_PASSWORD"
  pgHost <- lookupEnv "POSTGRES_HOST"
  -- https://hackage.haskell.org/package/postgresql-simple-0.7.0.0/docs/Database-PostgreSQL-Simple.html#t:ConnectInfo
  pure $ PSQL.defaultConnectInfo { PSQL.connectUser = fromMaybe "postgres" pgUser
                                 , PSQL.connectDatabase = fromMaybe "postgres" pgDb
                                 , PSQL.connectHost = fromMaybe "localhost" pgHost
                                 , PSQL.connectPassword = fromMaybe "postgres" pgPass }

setUpConn :: IO TestEnv
setUpConn = do
  connInfo <- getEnvConnectInfo
  conn <- PSQL.connect connInfo
  return $ TestEnv { conn, queue = testQueue }

dropConn :: TestEnv -> IO ()
dropConn (TestEnv { conn }) = do
  PSQL.close conn

withConn :: (TestEnv -> IO ()) -> IO ()
withConn = bracket setUpConn dropConn

withPGMQ :: (TestEnv -> IO ()) -> IO ()
withPGMQ f = withConn $ \testEnv -> bracket (setUpPGMQ testEnv) (tearDownPGMQ testEnv) (\_ -> f testEnv)
  where
    setUpPGMQ (TestEnv { conn, queue }) = do
      PGMQ.initialize conn
      PGMQ.dropQueue conn queue
      PGMQ.createQueue conn queue

    tearDownPGMQ (TestEnv { conn, queue }) _ = do
      PGMQ.dropQueue conn queue

      
pgmqSimpleTests :: Spec
pgmqSimpleTests = sequential $ aroundAll withPGMQ $ describe "PGMQ Simple" $ do
  -- it "can get metrics for non-existing queue" $ \(TestEnv { conn, queue }) -> do
  --   -- first of all, this should also work for non-existing queues
  --   metrics <- PGMQ.getMetrics conn queue
  --   metrics `shouldSatisfy` isNothing

  it "can get metrics for an empty queue" $ \(TestEnv { conn, queue }) -> do
    metrics <- PGMQ.getMetrics conn queue
    metrics `shouldSatisfy` isJust
    (PGMQ.queueLength <$> metrics) `shouldBe` (Just 0)

  it "listQueues properly returns our queue" $ \(TestEnv { conn, queue }) -> do
    queues <- PGMQ.listQueues conn
    ((\(PGMQ.QueueInfo { queueName }) -> queueName) <$> queues) `shouldContain` [queue]
    
