{ pkgs ? import ./nix/pinned.nix {} }:
let
  easy-ps = import ./nix/easy-ps.nix { inherit pkgs; };

  # sometimes bun is broken because of better-sqlite3
  fix-bun = pkgs.writeShellScriptBin "fix-bun" ''
    #!/usr/bin/env bash
    set -e

    cd ./node_modules/better-sqlite3
    bun install
  '';
  
  compile = pkgs.writeShellScriptBin "compile" ''
    #!/usr/bin/env bash
    set -e

    echo "Installing JS Dependencies"
    #yarn
    bun install
    bun install --dev

    echo "Compiling"
    bun run build
  '';
  
  build = pkgs.writeShellScriptBin "build" ''
    #!/usr/bin/env bash
    set -e

    # 0.15
    echo "Bundling"
    bun run bundle
  '';

    build-suggestions = pkgs.writeShellScriptBin "build-suggestions" ''
    #!/usr/bin/env bash
    set -e

    rm -rf ./output

    # 0.15
    echo "Bundling"
    bun spago bundle --module Main --outfile dist/bundle.js --purs-args "--json-errors" | jq -r '.warnings | map(select(.errorCode | test("UnusedDctorExplicitImport|UnusedDctorImport|UnusedDeclaration|UnusedExplicitImport|UnusedImport|UnusedName|UnusedTypeVar"))) | map(select(.filename | test("^.spago") | not)) | map("[\(.filename)::\(.position.startLine)] \(.message)\n \(.suggestion)") | join("\n----------\n")'
  '';


  build-css = pkgs.writeShellScriptBin "build-css" ''
    #!/usr/bin/env bash
    set -e

    #yarn css
    bun run css
  '';

  serve = pkgs.writeShellScriptBin "serve" ''
    #!/usr/bin/env bash
    set -e

    #yarn server
    bun run server

  '';

  build-watch = pkgs.writeShellScriptBin "build-watch" ''
    #!/usr/bin/env bash
    set -e

    echo "Build watch"
    bun spago build -w --then browserify
  '';

  build-zephyr = pkgs.writeShellScriptBin "build-zephyr" ''
    #!/usr/bin/env bash
    set -e

    bun spago build --purs-args '--codegen corefn,js'
    zephyr -f Main.main
    browserify-zephyr
  '';

  minify-bundle = pkgs.writeShellScriptBin "minify-bundle" ''
    #!/usr/bin/env bash
    set -e

    bun run minify
  '';

  repl = pkgs.writeShellScriptBin "repl" ''
    #!/usr/bin/env bash

    bun run repl
  '';

  test-ps = pkgs.writeShellScriptBin "test-ps" ''
    #!/usr/bin/env bash
    set -e

    compile

    echo "Testing"
    bun run test
  '';
  
  # TODO: Remove this when nixpkgs is updated to newer version (23.05
  # has nodejs 20.2.1 and we need >= 20.3.1 for crypto to work)
  #nodejs_20_3_1 = pkgs.callPackage ./nix/nodejs-v20.nix {};
in
pkgs.mkShell {
  buildInputs = [
    easy-ps.purs-0_15_15
    easy-ps.zephyr
    pkgs.ijq
    pkgs.jq
    #pkgs.esbuild
    pkgs.nodejs
    #pkgs.yarn
    pkgs.bun
    pkgs.sqlite

    # scripts
    build-css
    build-suggestions
    build-watch
    build-zephyr
    build
    compile
    fix-bun
    minify-bundle
    repl
    serve
    test-ps
  ];

  #shellHook = ''
  #  export PURS_IDE_SOURCES='${toString purs-project.unquotedSourceGlobs}'
  #'';
}

## how to build the project with nix dependencies:
#
# 1. start a nix shell (e.g. `nix-shell -j 20`, this uses 20 jobs to fetch deps)
# 2. run `yarn` to install npm deps
# 3a. run `build-purs` to build using nix store dependencies, and make sure to update your purescript ide tooling as necesssary
# 3b. or simply use `psc-package` as you might want to anyway
#
# note that the purescript compiler uses filepaths and timestamps, so using the above two commands
# interchangeably will lead to constant rebuilding of the entire project.
#
## how to update purs-packages.nix
#
# 1. run `nix/generate-packages-json.nix` to generate packages.json
# 2. run `nix/generate-purs-packages.nix` to generate purs-packages.nix
