{-|
Module      : Gargantext.Core.Config
Description : Textmining Collaborative Platform
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

Configuration for the gargantext server

-}

{-# LANGUAGE TemplateHaskell     #-}

module Gargantext.Core.Config (
    -- * Types
    GargConfig(..)

    -- * Lenses
  , gc_backend_name
  , gc_datafilepath
  , gc_epo_api_url
  , gc_frame_calc_url
  , gc_frame_istex_url
  , gc_frame_searx_url
  , gc_frame_visio_url
  , gc_frame_write_url
  , gc_js_id_timeout
  , gc_js_job_timeout
  , gc_masteruser
  , gc_max_docs_parsers
  , gc_max_docs_scrapers
  , gc_pubmed_api_key
  , gc_secretkey
  , gc_url
  , gc_url_backend_api
  , gc_frontend_config
  , gc_mail_config
  , gc_database_config
  , gc_nlp_config

  , mkProxyUrl
  ) where

import Data.Text as T
import Database.PostgreSQL.Simple qualified as PSQL
import Gargantext.Core.Config.Mail (MailConfig)
import Gargantext.Core.Config.NLP (NLPConfig)
import Gargantext.Core.Config.Types
import Gargantext.Prelude
import Servant.Client (BaseUrl(..), Scheme(Http), parseBaseUrl)
import Toml.Schema


-- | strip a given character from end of string
-- stripRight :: Char -> T.Text -> T.Text
-- stripRight c s = if T.last s == c then stripRight c (T.take (T.length s - 1) s) else s

data GargConfig = GargConfig { _gc_backend_name      :: !T.Text
                             , _gc_url               :: !T.Text
                             , _gc_url_backend_api   :: !T.Text

                             , _gc_masteruser        :: !T.Text
                             , _gc_secretkey         :: !T.Text

                             , _gc_datafilepath      :: !FilePath
                             -- , _gc_repofilepath      :: !FilePath

                             , _gc_frame_write_url   :: !T.Text
                             , _gc_frame_calc_url    :: !T.Text
                             , _gc_frame_visio_url   :: !T.Text

                             , _gc_frame_searx_url   :: !T.Text
                             , _gc_frame_istex_url   :: !T.Text

                             , _gc_max_docs_parsers  :: !Integer
                             , _gc_max_docs_scrapers :: !Integer

                             , _gc_js_job_timeout    :: !Integer
                             , _gc_js_id_timeout     :: !Integer

                             , _gc_pubmed_api_key    :: !T.Text

                             , _gc_epo_api_url       :: !T.Text

                             , _gc_frontend_config   :: !FrontendConfig
                             , _gc_mail_config       :: !MailConfig
                             , _gc_database_config   :: !PSQL.ConnectInfo
                             , _gc_nlp_config         :: !NLPConfig
                             }
  deriving (Generic, Show)

makeLenses ''GargConfig

instance FromValue GargConfig where
  fromValue = parseTableFromValue $ do
    _gc_frontend_config@(FrontendConfig { .. }) <- reqKey "frontend"
    _gc_mail_config <- reqKey "mail"
    db_config <- reqKey "database"
    _gc_nlp_config <- reqKey "nlp"
    secrets <- reqKey "secrets"
    _gc_datafilepath <- reqKeyOf "paths" $ parseTableFromValue $ reqKey "data_filepath"
    frames <- reqKeyOf "external" $ parseTableFromValue $ reqKey "frames"
    jobs <- reqKey "jobs"
    apis <- reqKey "apis"
    return $ GargConfig { _gc_backend_name = _fc_backend_name
                        , _gc_url          = _fc_url
                        , _gc_url_backend_api = _fc_url_backend_api
                        , _gc_masteruser = _s_master_user secrets
                        , _gc_secretkey = _s_secret_key secrets
                        , _gc_datafilepath
                        , _gc_frame_write_url = _f_write_url frames
                        , _gc_frame_calc_url = _f_calc_url frames
                        , _gc_frame_visio_url = _f_visio_url frames
                        , _gc_frame_searx_url = _f_searx_url frames
                        , _gc_frame_istex_url = _f_istex_url frames
                        , _gc_max_docs_parsers = _jc_max_docs_parsers jobs
                        , _gc_max_docs_scrapers = _jc_max_docs_scrapers jobs
                        , _gc_js_job_timeout = _jc_js_job_timeout jobs
                        , _gc_js_id_timeout = _jc_js_id_timeout jobs
                        , _gc_pubmed_api_key = _ac_pubmed_api_key apis
                        , _gc_epo_api_url = _ac_epo_api_url apis
                        , _gc_frontend_config
                        , _gc_mail_config
                        , _gc_database_config = unTOMLConnectInfo db_config
                        , _gc_nlp_config }



-- configCodec :: Toml.TomlCodec GargConfig
-- configCodec = GargConfig
--   <$> Toml.text "frontend.backend_name" .= _gc_backend_name
--   <*> (stripRight '/' <$> Toml.text "frontend.url") .= _gc_url
--   <*> (stripRight '/' <$> Toml.text "frontend.url_backend_api") .= _gc_url_backend_api
--   <*> Toml.text "secrets.master_user" .= _gc_masteruser
--   <*> Toml.text "secrets.secret_key" .= _gc_secretkey
--   <*> Toml.string "paths.data_filepath" .= _gc_datafilepath
--   <*> (stripRight '/' <$> Toml.text "external.frames.write_url") .= _gc_frame_write_url
--   <*> (stripRight '/' <$> Toml.text "external.frames.calc_url") .= _gc_frame_calc_url
--   <*> (stripRight '/' <$> Toml.text "external.frames.visio_url") .= _gc_frame_visio_url
--   <*> (stripRight '/' <$> Toml.text "external.frames.searx_url") .= _gc_frame_searx_url
--   <*> (stripRight '/' <$> Toml.text "external.frames.istex_url") .= _gc_frame_istex_url
--   <*> Toml.integer "jobs.max_docs_parsers" .= _gc_max_docs_parsers
--   <*> Toml.integer "jobs.max_docs_scrapers" .= _gc_max_docs_scrapers
--   <*> Toml.integer "jobs.js_job_timeout" .= _gc_js_job_timeout
--   <*> Toml.integer "jobs.js_id_timeout" .= _gc_js_id_timeout
--   <*> Toml.text "apis.pubmed.api_key" .= _gc_pubmed_api_key
--   <*> Toml.text "apis.epo.api_url" .= _gc_epo_api_url

  -- pure $ GargConfig
  --   { _gc_backend_name      = cs $ val' "BACKEND_NAME"
  --   , _gc_url               = stripRight '/' $ val' "URL"
  --   , _gc_url_backend_api   = stripRight '/' $ val' "URL_BACKEND_API"
  --   , _gc_masteruser        = val' "MASTER_USER"
  --   , _gc_secretkey         = val' "SECRET_KEY"
  --   , _gc_datafilepath      = cs $ val' "DATA_FILEPATH"
  --   , _gc_repofilepath      = cs $ val' "REPO_FILEPATH"
  --   , _gc_frame_write_url   = stripRight '/' $ val' "FRAME_WRITE_URL"
  --   , _gc_frame_calc_url    = stripRight '/' $ val' "FRAME_CALC_URL"
  --   , _gc_frame_visio_url   = stripRight '/' $ val' "FRAME_VISIO_URL"
  --   , _gc_frame_searx_url   = stripRight '/' $ val' "FRAME_SEARX_URL"
  --   , _gc_frame_istex_url   = stripRight '/' $ val' "FRAME_ISTEX_URL"
  --   , _gc_max_docs_parsers  = read $ cs $ val' "MAX_DOCS_PARSERS"
  --   , _gc_max_docs_scrapers = read $ cs $ val' "MAX_DOCS_SCRAPERS"
  --   , _gc_pubmed_api_key    = val' "PUBMED_API_KEY"
  --   , _gc_js_job_timeout    = read $ cs $ val' "JS_JOB_TIMEOUT"
  --   , _gc_js_id_timeout     = read $ cs $ val' "JS_ID_TIMEOUT"
  --   , _gc_epo_api_url       = cs $ val' "EPO_API_URL"
  --   }



mkProxyUrl :: GargConfig -> MicroServicesSettings -> BaseUrl
mkProxyUrl GargConfig{..} MicroServicesSettings{..} =
  case parseBaseUrl (T.unpack _gc_url) of
    Nothing -> BaseUrl Http "localhost" 80 ""
    Just bh -> bh { baseUrlPort = _msProxyPort }
