{-|
Module      : Gargantext.Core.Config
Description : Textmining Collaborative Platform
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

Configuration for the gargantext server

-}

{-# LANGUAGE TemplateHaskell     #-}

module Gargantext.Core.Config (
    -- * Types
    GargConfig(..)

    -- * Lenses
  , gc_backend_name
  , gc_datafilepath
  , gc_epo_api_url
  , gc_js_id_timeout
  , gc_js_job_timeout
  , gc_masteruser
  , gc_max_docs_parsers
  , gc_max_docs_scrapers
  , gc_pubmed_api_key
  , gc_secretkey
  , gc_url
  , gc_url_backend_api
  , gc_frontend_config
  , gc_mail_config
  , gc_database_config
  , gc_nlp_config
  , gc_notifications_config
  , gc_frames

  , mkProxyUrl
  ) where

import Data.Text as T
import Database.PostgreSQL.Simple qualified as PSQL
import Gargantext.Core.Config.Mail (MailConfig)
import Gargantext.Core.Config.NLP (NLPConfig)
import Gargantext.Core.Config.Types
import Gargantext.Prelude
import Servant.Client (BaseUrl(..), Scheme(Http), parseBaseUrl)
import Toml.Schema


-- | strip a given character from end of string
-- stripRight :: Char -> T.Text -> T.Text
-- stripRight c s = if T.last s == c then stripRight c (T.take (T.length s - 1) s) else s

data GargConfig = GargConfig { _gc_backend_name         :: !T.Text
                             , _gc_url                  :: !T.Text
                             , _gc_url_backend_api      :: !T.Text

                             , _gc_masteruser           :: !T.Text
                             , _gc_secretkey            :: !T.Text

                             , _gc_datafilepath         :: !FilePath
                             -- , _gc_repofilepath      :: !FilePath

                             , _gc_max_docs_parsers     :: !Integer
                             , _gc_max_docs_scrapers    :: !Integer

                             , _gc_js_job_timeout       :: !Integer
                             , _gc_js_id_timeout        :: !Integer

                             , _gc_pubmed_api_key       :: !T.Text

                             , _gc_epo_api_url          :: !T.Text

                             , _gc_frontend_config      :: !FrontendConfig
                             , _gc_mail_config          :: !MailConfig
                             , _gc_database_config      :: !PSQL.ConnectInfo
                             , _gc_nlp_config           :: !NLPConfig
                             , _gc_notifications_config :: !NotificationsConfig
                             , _gc_frames               :: !FramesConfig
                             }
  deriving (Generic, Show)

makeLenses ''GargConfig

instance FromValue GargConfig where
  fromValue = parseTableFromValue $ do
    _gc_frontend_config@(FrontendConfig { .. }) <- reqKey "frontend"
    _gc_mail_config <- reqKey "mail"
    db_config <- reqKey "database"
    _gc_nlp_config <- reqKey "nlp"
    secrets <- reqKey "secrets"
    _gc_datafilepath <- reqKeyOf "paths" $ parseTableFromValue $ reqKey "data_filepath"
    _gc_frames <- reqKeyOf "external" $ parseTableFromValue $ reqKey "frames"
    jobs <- reqKey "jobs"
    apis <- reqKey "apis"
    _gc_notifications_config <- reqKey "notifications"
    return $ GargConfig { _gc_backend_name = _fc_backend_name
                        , _gc_url          = _fc_url
                        , _gc_url_backend_api = _fc_url_backend_api
                        , _gc_masteruser = _s_master_user secrets
                        , _gc_secretkey = _s_secret_key secrets
                        , _gc_datafilepath
                        , _gc_max_docs_parsers = _jc_max_docs_parsers jobs
                        , _gc_max_docs_scrapers = _jc_max_docs_scrapers jobs
                        , _gc_js_job_timeout = _jc_js_job_timeout jobs
                        , _gc_js_id_timeout = _jc_js_id_timeout jobs
                        , _gc_pubmed_api_key = _ac_pubmed_api_key apis
                        , _gc_epo_api_url = _ac_epo_api_url apis
                        , _gc_frontend_config
                        , _gc_mail_config
                        , _gc_database_config = unTOMLConnectInfo db_config
                        , _gc_nlp_config
                        , _gc_notifications_config
                        , _gc_frames }
instance ToValue GargConfig where
  toValue = defaultTableToValue
instance ToTable GargConfig where
  toTable (GargConfig { .. }) =
    table [ "frontend" .= _gc_frontend_config
          , "secrets" .= secrets
          , "paths" .= table [ "data_filepath" .= _gc_datafilepath ]
          , "apis" .= apis
          , "external" .= table [ "frames" .= _gc_frames ]
          , "jobs" .= jobs
          , "database" .= TOMLConnectInfo _gc_database_config
          , "mail" .= _gc_mail_config
          , "notifications" .= _gc_notifications_config
          , "nlp" .= _gc_nlp_config
          ]
    where
      secrets = SecretsConfig { _s_master_user = _gc_masteruser
                              , _s_secret_key = _gc_secretkey }
      apis = APIsConfig { _ac_pubmed_api_key = _gc_pubmed_api_key
                        , _ac_epo_api_url = _gc_epo_api_url }
      jobs = JobsConfig { _jc_max_docs_parsers = _gc_max_docs_parsers
                        , _jc_max_docs_scrapers = _gc_max_docs_scrapers
                        , _jc_js_job_timeout = _gc_js_job_timeout
                        , _jc_js_id_timeout = _gc_js_id_timeout }


mkProxyUrl :: GargConfig -> MicroServicesSettings -> BaseUrl
mkProxyUrl GargConfig{..} MicroServicesSettings{..} =
  case parseBaseUrl (T.unpack _gc_url) of
    Nothing -> BaseUrl Http "localhost" 80 ""
    Just bh -> bh { baseUrlPort = _msProxyPort }
