{-|
Module      : Gargantext.Database.Action.Node
Description :
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX
-}


{-# OPTIONS_GHC -fno-warn-orphans        #-}

{-# LANGUAGE Arrows                 #-}
{-# LANGUAGE ConstraintKinds        #-}
{-# LANGUAGE FunctionalDependencies #-}
{-# LANGUAGE TemplateHaskell        #-}
{-# LANGUAGE TypeFamilies           #-}

module Gargantext.Database.Action.Node
  where

import Control.Lens (view)
import Data.Text qualified as T
import Gargantext.API.Admin.Types (settings, _microservicesSettings, HasSettings)
import Gargantext.Core
import Gargantext.Core.Config (GargConfig(..), mkProxyUrl)
import Gargantext.Core.Config.Types (FramesConfig(..), MicroServicesSettings(..))
import Gargantext.Core.Types (Name)
import Gargantext.Database.Admin.Types.Hyperdata
import Gargantext.Database.Admin.Types.Hyperdata.Default
import Gargantext.Database.Admin.Types.Node
import Gargantext.Database.Prelude (HasConfig(..), DBCmd')
import Gargantext.Database.Query.Table.Node
import Gargantext.Database.Query.Table.Node.Error
import Gargantext.Database.Query.Table.Node.UpdateOpaleye (updateHyperdata)
import Gargantext.Prelude hiding (hash)
import Gargantext.Prelude.Crypto.Hash (hash)
import Servant.Client.Core.BaseUrl

------------------------------------------------------------------------
-- | TODO mk all others nodes
mkNodeWithParent :: (HasNodeError err, HasDBid NodeType, HasSettings env)
                 => NodeType
                 -> Maybe ParentId
                 -> UserId
                 -> Name
                 -> DBCmd' env err [NodeId]
mkNodeWithParent NodeUser (Just pId) uid  _ = nodeError $ NodeCreationFailed $ UserParentAlreadyExists uid pId

------------------------------------------------------------------------
-- | MkNode, insert and eventually configure Hyperdata
mkNodeWithParent NodeUser Nothing uId name =
  insertNodesWithParentR Nothing [node NodeUser name defaultHyperdataUser Nothing uId]

mkNodeWithParent _ Nothing uId _ = nodeError $ NodeCreationFailed $ UserParentDoesNotExist uId
------------------------------------------------------------------------
mkNodeWithParent Notes i u n =
  mkNodeWithParent_ConfigureHyperdata Notes i u n

mkNodeWithParent Calc i u n =
  mkNodeWithParent_ConfigureHyperdata Calc i u n

mkNodeWithParent NodeFrameVisio i u n =
  mkNodeWithParent_ConfigureHyperdata NodeFrameVisio i u n

mkNodeWithParent NodeFrameNotebook i u n =
  mkNodeWithParent_ConfigureHyperdata NodeFrameNotebook i u n



mkNodeWithParent nt (Just pId) uId name  = (:[]) <$> insertNode nt (Just name) Nothing pId uId
-- mkNodeWithParent _ _ _ _ = errorWith "[G.D.A.Node.mkNodeWithParent] nees parent"


-- | Sugar to create a node, get its NodeId and update its Hyperdata after
mkNodeWithParent_ConfigureHyperdata :: (HasNodeError err, HasDBid NodeType, HasSettings env)
                                    => NodeType
                                    -> Maybe ParentId
                                    -> UserId
                                    -> Name
                                    -> DBCmd' env err [NodeId]
mkNodeWithParent_ConfigureHyperdata Notes (Just i) uId name =
  mkNodeWithParent_ConfigureHyperdata' Notes (Just i) uId name

mkNodeWithParent_ConfigureHyperdata Calc (Just i) uId name =
  mkNodeWithParent_ConfigureHyperdata' Calc (Just i) uId name

mkNodeWithParent_ConfigureHyperdata NodeFrameVisio (Just i) uId name =
  mkNodeWithParent_ConfigureHyperdata' NodeFrameVisio (Just i) uId name

mkNodeWithParent_ConfigureHyperdata NodeFrameNotebook (Just i) uId name = (:[]) <$>
  insertNode NodeFrameNotebook  (Just "Notebook")
                                (Just $ DefaultFrameCode $ HyperdataFrame { _hf_base = "Codebook"
                                                                          , _hf_frame_id = name }) i uId

mkNodeWithParent_ConfigureHyperdata    _ _ _ _ = nodeError NotImplYet

-- | Creates the base URL for the notes microservices proxy, or defaults
-- to the notes microservice if the proxy has been disabled from the settings.
internalNotesProxy :: GargConfig -> MicroServicesSettings -> T.Text
internalNotesProxy cfg msSettings
  | _msProxyEnabled msSettings = T.pack $ showBaseUrl proxyUrl <> "/notes"
  | otherwise                  = _f_write_url $ _gc_frames cfg
  where
    proxyUrl = mkProxyUrl cfg msSettings

-- | Function not exposed
mkNodeWithParent_ConfigureHyperdata' :: (HasNodeError err, HasDBid NodeType, HasSettings env)
                                     => NodeType
                                     -> Maybe ParentId
                                     -> UserId
                                     -> Name
                                     -> DBCmd' env err [NodeId]
mkNodeWithParent_ConfigureHyperdata' nt (Just i) uId name = do
  nodeId <- case nt of
     Notes -> insertNode Notes (Just name)   Nothing i uId
     Calc  -> insertNode Calc  (Just name)   Nothing i uId
     NodeFrameVisio -> insertNode NodeFrameVisio (Just name)   Nothing i uId
     _              -> nodeError NeedsConfiguration

  cfg <- view hasConfig
  stt <- view settings
  u <- case nt of
        Notes -> pure $ internalNotesProxy cfg (_microservicesSettings stt)
        Calc  -> pure $ _f_calc_url $ _gc_frames cfg
        NodeFrameVisio -> pure $ _f_visio_url $ _gc_frames cfg
        _              -> nodeError NeedsConfiguration
  let
    s = _gc_secretkey cfg
    hd = HyperdataFrame u (hash $ s <> (show nodeId))
  _ <- updateHyperdata nodeId hd
  pure [nodeId]
mkNodeWithParent_ConfigureHyperdata' _ Nothing uId _ = nodeError $ NodeCreationFailed $ UserParentDoesNotExist uId
