{-|
Module      : Gargantext.Core.AsyncUpdates.CentralExchange
Description : Central exchange (asynchronous notifications)
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

https://gitlab.iscpif.fr/gargantext/haskell-gargantext/issues/341

Docs:
https://dev.sub.gargantext.org/#/share/Notes/187918
    
-}

module Gargantext.Core.AsyncUpdates.CentralExchange where

-- import Control.Concurrent (threadDelay)
import Control.Concurrent.Async qualified as Async
import Data.Aeson qualified as Aeson
import Data.ByteString.Char8 qualified as C
import Data.ByteString.Lazy qualified as BSL
import Gargantext.Core.AsyncUpdates.CentralExchange.Types
import Gargantext.Core.AsyncUpdates.Constants (cePort, dispatcherInternalPort)
import Gargantext.Prelude
import Nanomsg
    
{-

Central exchange is a service, which gathers messages from various
places and informs the Dispatcher (which will then inform users about
various events).
    
-}

gServer :: IO ()
gServer = do
    withSocket Pull $ \s -> do
      withSocket Push $ \s_dispatcher -> do
        _ <- bind s ("tcp://*:" <> show cePort)
        _ <- connect s_dispatcher ("tcp://localhost:" <> show dispatcherInternalPort)
        forever $ do
          putText "[central_exchange] receiving"
          r <- recv s
          C.putStrLn r
          case Aeson.decode (BSL.fromStrict r) of
            Just (UpdateTreeFirstLevel node_id) -> do
              putText $ "[central_exchange] update tree: " <> show node_id
              putText $ "[central_exchange] sending that to the dispatcher: " <> show node_id
              -- To make this more robust, use withAsync so we don't
              -- block the main thread (send is blocking)
              
              -- NOTE: If we're flooded with messages, and send is
              -- slow, we might be spawning many threads...

              -- NOTE: Currently we just forward the message that we
              -- got. So in theory central exchange isn't needed (we
              -- could ping dispatcher directly). However, I think
              -- it's better to have this as a separate
              -- component. Currently I built this inside
              -- gargantext-server but maybe it can be a separate
              -- process, independent of the server.
              Async.withAsync (pure ()) $ \_ -> do
                send s_dispatcher r
            _ -> putText "[central_exchange] unknown"
  

notify :: CEMessage -> IO ()
notify ceMessage = do
  withSocket Push $ \s -> do
    _ <- connect s ("tcp://localhost:" <> show cePort)
    let str = Aeson.encode ceMessage
    send s $ BSL.toStrict str
