{-|
Module      : Gargantext.Core.AsyncUpdates.Dispatcher
Description : Dispatcher (handles websocket connections, accepts message from central exchange)
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

https://gitlab.iscpif.fr/gargantext/haskell-gargantext/issues/341

Docs:
https://dev.sub.gargantext.org/#/share/Notes/187918
    
-}

{-# LANGUAGE TypeOperators #-}
{-# OPTIONS_GHC -fno-warn-unused-matches -fno-warn-unused-imports #-}
    
module Gargantext.Core.AsyncUpdates.Dispatcher where

import Control.Concurrent.Async qualified as Async
import Data.Aeson ((.:), (.=))
import Data.Aeson qualified as Aeson
import Data.Aeson.Types (prependFailure, typeMismatch)
import Data.ByteString.Char8 qualified as C
import Data.ByteString.Lazy qualified as BSL
import Data.List (nubBy)
import Data.UUID.V4 as UUID
import Gargantext.API.Admin.Auth.Types (AuthenticatedUser(_auth_user_id))
import Gargantext.API.Admin.Types (jwtSettings, Settings, jwtSettings)
import Gargantext.Core.AsyncUpdates.CentralExchange.Types qualified as CETypes
import Gargantext.Core.Types (NodeId, UserId)
import Gargantext.Prelude
import GHC.Conc (TVar, newTVarIO, readTVar, writeTVar)
import Nanomsg
import Network.WebSockets qualified as WS
import Servant
import Servant.API.WebSocket qualified as WS
import Servant.Auth.Server (verifyJWT)

{-

Dispatcher is a service, which provides couple of functionalities:
- handles WebSocket connections and manages them
- accepts messages from central exchange
- dispatches these messages to connected users
    
-}

data Topic =
  -- | Update given Servant Job (we currently send a request every
  -- | second to get job status).
  --  UpdateJob JobID
  -- | Given parent node id, trigger update of the node and its
  --   children (e.g. list is automatically created in a corpus)
  UpdateTree NodeId
  deriving (Eq, Show)

instance FromJSON Topic where
  parseJSON = Aeson.withObject "Topic" $ \o -> do
    type_ <- o .: "type"
    case type_ of
      "update_tree" -> do
        node_id <- o .: "node_id"
        pure $ UpdateTree node_id
      s -> prependFailure "parsing type failed, " (typeMismatch "type" s)

instance ToJSON Topic where
  toJSON (UpdateTree node_id) = Aeson.object [
      "type" .= toJSON ("update_tree" :: Text)
    , "node_id" .= toJSON node_id
    ]
    
data ConnectedUser =
    CUUser UserId
  | CUPublic
  deriving (Eq, Show)

newtype WSKeyConnection = WSKeyConnection (ByteString, WS.Connection)
eqWSKeyConnection :: WSKeyConnection -> WSKeyConnection -> Bool
eqWSKeyConnection ws1 ws2 = wsKey ws1 == wsKey ws2
showWSKeyConnection :: WSKeyConnection -> Text
showWSKeyConnection ws = "WSKeyConnection " <> show (wsKey ws)
wsKey :: WSKeyConnection -> ByteString
wsKey (WSKeyConnection (key, _conn)) = key
wsConn :: WSKeyConnection -> WS.Connection
wsConn (WSKeyConnection (_key, conn)) = conn
    
data Subscription =
  Subscription {
      s_connected_user    :: ConnectedUser
    , s_ws_key_connection :: WSKeyConnection
    , s_topic             :: Topic }
eqSub :: Subscription -> Subscription -> Bool
eqSub sub1 sub2 =
  s_connected_user sub1 == s_connected_user sub2 &&
  s_ws_key_connection sub2 `eqWSKeyConnection` s_ws_key_connection sub2 &&
  s_topic sub1 == s_topic sub2
showSub :: Subscription -> Text
showSub sub =
  "Subscription " <> show (s_connected_user sub) <>
  " " <> showWSKeyConnection (s_ws_key_connection sub) <>
  " " <> show (s_topic sub)
subKey :: Subscription -> ByteString
subKey sub = wsKey $ s_ws_key_connection sub


type Token = Text
    
{-
We accept requests for subscription/unsubscription via websocket.

We could instead handle 1 websocket connection per every topic
subscription (e.g. parse headers in WS.PendingConnection. However, WS
by default can handle 65k concurrent connections. With multiple users
having multiple components open, we could exhaust that limit quickly.

Hence, we architect this to have 1 websocket connection per web
browser.
-}
data WSRequest =
    WSSubscribe Topic
  | WSUnsubscribe Topic
  | WSAuthorize Token
  | WSDeauthorize
  deriving (Eq, Show)
instance FromJSON WSRequest where
  parseJSON = Aeson.withObject "WSRequest" $ \o -> do
    request <- o .: "request"
    case request of
      "subscribe" -> do
        topic <- o .: "topic"
        pure $ WSSubscribe topic
      "unsubscribe" -> do
        topic <- o .: "topic"
        pure $ WSUnsubscribe topic
      "authorize" -> do
         token <- o .: "token"
         pure $ WSAuthorize token
      "deauthorize" -> pure $ WSDeauthorize
      s -> prependFailure "parsing request type failed, " (typeMismatch "request" s)
    
data Dispatcher =
 Dispatcher { d_subscriptions :: TVar [Subscription]
            , d_ws_server     :: Server WSAPI
            , d_ce_listener   :: ThreadId
            }
    

dispatcher :: Settings -> IO Dispatcher
dispatcher authSettings = do
  subscriptions <- newTVarIO ([] :: [Subscription])

  let server = wsServer authSettings subscriptions

  d_ce_listener <- forkIO (ce_listener subscriptions)

  pure $ Dispatcher { d_subscriptions = subscriptions
                    , d_ws_server     = server
                    , d_ce_listener   = d_ce_listener }


-- | TODO Allow only 1 topic subscription per connection. It doesn't
-- | make sense to send multiple notifications of the same type to the
-- | same connection.
insertSubscription :: TVar [Subscription] -> Subscription -> IO [Subscription]
insertSubscription subscriptions sub =
  atomically $ do
    s <- readTVar subscriptions
    let ss = nubBy eqSub $ s <> [sub]
    writeTVar subscriptions ss
    pure ss
     
removeSubscription :: TVar [Subscription] -> Subscription -> IO [Subscription]
removeSubscription subscriptions sub =
  atomically $ do
    s <- readTVar subscriptions
    let ss = filter (\sub' -> not $ sub `eqSub` sub') s
    writeTVar subscriptions ss
    pure ss
   
removeSubscriptionsForWSKey :: TVar [Subscription] -> WSKeyConnection -> IO [Subscription]
removeSubscriptionsForWSKey subscriptions ws =
  atomically $ do
    s <- readTVar subscriptions
    let ss = filter (\sub -> subKey sub /= wsKey ws) s
    writeTVar subscriptions ss
    pure ss
    
type WSAPI = "ws" :> WS.WebSocketPending

wsServer :: Settings -> TVar [Subscription] -> Server WSAPI
wsServer authSettings subscriptions = streamData
 where
  streamData :: MonadIO m => WS.PendingConnection -> m ()
  streamData pc = do
    let reqHead = WS.pendingRequest pc
    -- WebSocket specification says that a pending request should send
    -- some unique, Sec-WebSocket-Key string. We use this to compare
    -- connections (WS.Connection doesn't implement an Eq instance).
    let mKey = head $ filter (\(k, _) -> k == "Sec-WebSocket-Key") $ WS.requestHeaders reqHead
    let key' = snd $ fromMaybe (panicTrace "Sec-WebSocket-Key not found!") mKey
    -- Unfortunately, a single browsers sends the same
    -- Sec-WebSocket-Key so we want to make that even more unique.
    uuid <- liftIO $ UUID.nextRandom
    let key = key' <> "-" <> show uuid
    putText $ show $ WS.requestHeaders reqHead
    c <- liftIO $ WS.acceptRequest pc
    let ws = WSKeyConnection (key, c)
    _ <- liftIO $ Async.concurrently (wsLoop ws) (pingLoop ws)
    -- _ <- liftIO $ Async.withAsync (pure ()) (\_ -> wsLoop ws)
    pure ()

  -- | Send a ping control frame periodically, otherwise the
  -- | connection is dropped. NOTE that 'onPing' message is not
  -- | supported in the JS API: either the browser supports this or
  -- | not:
  -- | https://stackoverflow.com/questions/10585355/sending-websocket-ping-pong-frame-from-browser
  pingLoop ws = do
    forever $ do
      -- WS.sendDataMessage (wsConn ws) (WS.Text (Aeson.encode Ping) Nothing)
      WS.sendPing (wsConn ws) ("" :: Text)
      threadDelay $ 10 * 1000000
    
  wsLoop ws = flip finally disconnect $ do
      putText "[wsLoop] connecting"
      wsLoop' CUPublic

    where
      wsLoop' user = do
        dm <- WS.receiveDataMessage (wsConn ws)
        
        newUser <- case dm of
          WS.Text dm' _ -> do
            case Aeson.decode dm' of
              Nothing -> do
                putText "[wsLoop] unknown message"
                return user
              Just (WSSubscribe topic) -> do
                -- TODO Fix s_connected_user based on header
                let sub = Subscription { s_connected_user = user
                                       , s_ws_key_connection = ws
                                       , s_topic = topic }
                ss <- insertSubscription subscriptions sub
                putText $ "[wsLoop] subscriptions: " <> show (showSub <$> ss)
                return user
              Just (WSUnsubscribe topic) -> do
                -- TODO Fix s_connected_user based on header
                let sub = Subscription { s_connected_user = CUPublic
                                      , s_ws_key_connection = ws
                                      , s_topic = topic }
                ss <- removeSubscription subscriptions sub
                putText $ "[wsLoop] subscriptions: " <> show (showSub <$> ss)
                return user
              Just (WSAuthorize token) -> do
                let jwtS = authSettings ^. jwtSettings
                mUser <- liftBase $ verifyJWT jwtS (encodeUtf8 token)

                putText $ "[wsLoop] authorized user: " <> show mUser

                -- TODO Update my subscriptions!

                return $ fromMaybe user (CUUser . _auth_user_id <$> mUser)
              Just WSDeauthorize -> do
                -- TODO Update my subscriptions!
                pure CUPublic
          _ -> do
            putText "[wsLoop] binary ws messages not supported"
            return user
          
        wsLoop' newUser

      disconnect = do
        putText "[wsLoop] disconnecting..."
        ss <- removeSubscriptionsForWSKey subscriptions ws
        putText $ "[wsLoop] subscriptions: " <> show (showSub <$> ss)


data Notification =
   Notification Topic
  deriving (Eq, Show)

instance ToJSON Notification where
  toJSON (Notification topic) = Aeson.object [
      "notification" .= toJSON topic
    ]
    

ce_listener :: TVar [Subscription] -> IO ()
ce_listener subscriptions = do
  withSocket Pull $ \s -> do
    _ <- bind s "tcp://*:5561"
    forever $ do
      putText "[ce_listener] receiving"
      r <- recv s
      C.putStrLn r
      case Aeson.decode (BSL.fromStrict r) of
        Nothing -> putText "[ce_listener] unknown message from central exchange"
        Just ceMessage -> do
          subs <- atomically $ readTVar subscriptions
          -- TODO This isn't safe: we atomically fetch subscriptions,
          -- then send notifications one by one. In the meantime, a
          -- subscription could end or new ones could appear
          let filteredSubs = filterCEMessageSubs ceMessage subs
          mapM_ (sendNotification ceMessage) filteredSubs
  where
    sendNotification :: CETypes.CEMessage -> Subscription -> IO ()
    sendNotification ceMessage sub = do
      let ws = s_ws_key_connection sub
      -- send the same thing to everyone for now
      WS.sendDataMessage (wsConn ws) (WS.Text (Aeson.encode $ Notification $ s_topic sub) Nothing)


-- Custom filtering of list of Subscriptions based on
-- CETypes.CEMessage.
-- For example, we can add CEMessage.Broadcast to propagate a
-- notification to all connections.
filterCEMessageSubs :: CETypes.CEMessage -> [Subscription] -> [Subscription]
filterCEMessageSubs (CETypes.UpdateTreeFirstLevel node_id) subscriptions =
  filter (\sub -> s_topic sub == UpdateTree node_id) subscriptions
