{-|
Module      : Gargantext.Core.AsyncUpdates.Dispatcher
Description : Dispatcher (handles websocket connections, accepts message from central exchange)
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

https://gitlab.iscpif.fr/gargantext/haskell-gargantext/issues/341

Docs:
https://dev.sub.gargantext.org/#/share/Notes/187918
    
-}

{-# LANGUAGE TypeOperators #-}
{-# OPTIONS_GHC -fno-warn-unused-matches -fno-warn-unused-imports #-}
    
module Gargantext.Core.AsyncUpdates.Dispatcher where

import Control.Concurrent.Async qualified as Async
import Control.Lens (view)
import Data.Aeson ((.:), (.=))
import Data.Aeson qualified as Aeson
import Data.Aeson.Types (prependFailure, typeMismatch)
import Data.ByteString.Char8 qualified as C
import Data.ByteString.Lazy qualified as BSL
import Data.List (nubBy)
import DeferredFolds.UnfoldlM qualified as UnfoldlM
import Data.UUID.V4 as UUID
import Gargantext.API.Admin.Auth.Types (AuthenticatedUser(_auth_user_id))
import Gargantext.API.Admin.EnvTypes (env_dispatcher)
import Gargantext.API.Admin.Types (jwtSettings, Settings, jwtSettings)
import Gargantext.API.Prelude (IsGargServer)
import Gargantext.Core.AsyncUpdates.CentralExchange.Types qualified as CETypes
import Gargantext.Core.AsyncUpdates.Constants as AUConstants
import Gargantext.Core.AsyncUpdates.Dispatcher.Types
import Gargantext.Core.Types (NodeId, UserId)
import Gargantext.Prelude
import GHC.Conc (TVar, newTVarIO, readTVar, writeTVar)
import Nanomsg
import Network.WebSockets qualified as WS
import Protolude.Base (Show(showsPrec))
import Servant
-- import Servant.API.NamedRoutes ((:-))
import Servant.API.WebSocket qualified as WS
import Servant.Auth.Server (verifyJWT)
import Servant.Server.Generic (AsServer, AsServerT)
import StmContainers.Set as SSet

{-

Dispatcher is a service, which provides couple of functionalities:
- handles WebSocket connections and manages them
- accepts messages from central exchange
- dispatches these messages to connected users
    
-}
    

dispatcher :: Settings -> IO Dispatcher
dispatcher authSettings = do
  subscriptions <- SSet.newIO

  -- let server = wsServer authSettings subscriptions

  d_ce_listener <- forkIO (dispatcher_listener subscriptions)

  pure $ Dispatcher { d_subscriptions = subscriptions
                    -- , d_ws_server     = server
                    , d_ce_listener   = d_ce_listener }


-- | TODO Allow only 1 topic subscription per connection. It doesn't
-- | make sense to send multiple notifications of the same type to the
-- | same connection.
insertSubscription :: SSet.Set Subscription -> Subscription -> IO ()
insertSubscription subscriptions sub = do
  atomically $ SSet.insert sub subscriptions
    -- s <- readTVar subscriptions
    -- let ss = nubBy eqSub $ s <> [sub]
    -- writeTVar subscriptions ss
    -- -- pure ss
    -- pure ()
     
removeSubscription :: SSet.Set Subscription -> Subscription -> IO ()
removeSubscription subscriptions sub = do
  atomically $ SSet.delete sub subscriptions
    -- s <- readTVar subscriptions
    -- let ss = filter (\sub' -> not $ sub `eqSub` sub') s
    -- writeTVar subscriptions ss
    -- pure ss
   
removeSubscriptionsForWSKey :: SSet.Set Subscription -> WSKeyConnection -> IO ()
removeSubscriptionsForWSKey subscriptions ws = do
  atomically $ do
    let toDelete = UnfoldlM.filter (\sub -> return $ subKey sub == wsKey ws) $ SSet.unfoldlM subscriptions
    UnfoldlM.mapM_ (\sub -> SSet.delete sub subscriptions) toDelete
  -- atomically $ do
  --   s <- readTVar subscriptions
  --   let ss = filter (\sub -> subKey sub /= wsKey ws) s
  --   writeTVar subscriptions ss
  --   pure ss
    
newtype WSAPI mode = WSAPI {
    wsAPI :: mode :- "ws" :> WS.WebSocketPending
  } deriving Generic

-- wsServer :: IsGargServer env err m => Settings -> SSet.Set Subscription -> WSAPI (AsServerT m)
-- wsServer authSettings subscriptions = WSAPI { wsAPI = streamData }
wsServer :: IsGargServer env err m => Settings -> WSAPI (AsServerT m)
wsServer authSettings = WSAPI { wsAPI = streamData }
  where
    streamData :: IsGargServer env err m => WS.PendingConnection -> m ()
    streamData pc = do
      d <- view env_dispatcher
      let subscriptions = d_subscriptions d
      let reqHead = WS.pendingRequest pc
      -- WebSocket specification says that a pending request should send
      -- some unique, Sec-WebSocket-Key string. We use this to compare
      -- connections (WS.Connection doesn't implement an Eq instance).
      let mKey = head $ filter (\(k, _) -> k == "Sec-WebSocket-Key") $ WS.requestHeaders reqHead
      let key' = snd $ fromMaybe (panicTrace "Sec-WebSocket-Key not found!") mKey
      -- Unfortunately, a single browsers sends the same
      -- Sec-WebSocket-Key so we want to make that even more unique.
      uuid <- liftBase $ UUID.nextRandom
      let key = key' <> "-" <> show uuid
      liftBase $ putText $ show $ WS.requestHeaders reqHead
      c <- liftBase $ WS.acceptRequest pc
      let ws = WSKeyConnection (key, c)
      _ <- liftBase $ Async.concurrently (wsLoop subscriptions ws) (pingLoop ws)
      -- _ <- liftIO $ Async.withAsync (pure ()) (\_ -> wsLoop ws)
      pure ()
  
    -- | Send a ping control frame periodically, otherwise the
    -- | connection is dropped. NOTE that 'onPing' message is not
    -- | supported in the JS API: either the browser supports this or
    -- | not:
    -- | https://stackoverflow.com/questions/10585355/sending-websocket-ping-pong-frame-from-browser
    pingLoop ws = do
      forever $ do
        -- WS.sendDataMessage (wsConn ws) (WS.Text (Aeson.encode Ping) Nothing)
        WS.sendPing (wsConn ws) ("" :: Text)
        threadDelay $ 10 * 1000000
      
    wsLoop subscriptions ws = flip finally disconnect $ do
      putText "[wsLoop] connecting"
      wsLoop' CUPublic
  
      where
        wsLoop' user = do
          dm <- WS.receiveDataMessage (wsConn ws)
          
          newUser <- case dm of
            WS.Text dm' _ -> do
              case Aeson.decode dm' of
                Nothing -> do
                  putText "[wsLoop] unknown message"
                  return user
                Just (WSSubscribe topic) -> do
                  -- TODO Fix s_connected_user based on header
                  let sub = Subscription { s_connected_user = user
                                         , s_ws_key_connection = ws
                                         , s_topic = topic }
                  _ss <- insertSubscription subscriptions sub
                  -- putText $ "[wsLoop] subscriptions: " <> show (showSub <$> ss)
                  return user
                Just (WSUnsubscribe topic) -> do
                  let sub = Subscription { s_connected_user = user
                                         , s_ws_key_connection = ws
                                         , s_topic = topic }
                  _ss <- removeSubscription subscriptions sub
                  -- putText $ "[wsLoop] subscriptions: " <> show (showSub <$> ss)
                  return user
                Just (WSAuthorize token) -> do
                  let jwtS = authSettings ^. jwtSettings
                  mUser <- liftBase $ verifyJWT jwtS (encodeUtf8 token)
  
                  putText $ "[wsLoop] authorized user: " <> show mUser
  
                  -- TODO Update my subscriptions!
  
                  return $ fromMaybe user (CUUser . _auth_user_id <$> mUser)
                Just WSDeauthorize -> do
                  -- TODO Update my subscriptions!
                  pure CUPublic
            _ -> do
              putText "[wsLoop] binary ws messages not supported"
              return user
            
          wsLoop' newUser
  
        disconnect = do
          putText "[wsLoop] disconnecting..."
          _ss <- removeSubscriptionsForWSKey subscriptions ws
          -- putText $ "[wsLoop] subscriptions: " <> show (show <$> ss)
          return ()

          
dispatcher_listener :: SSet.Set Subscription -> IO ()
dispatcher_listener subscriptions = do
  withSocket Pull $ \s -> do
    _ <- bind s ("tcp://*:" <> show AUConstants.dispatcherInternalPort)
    forever $ do
      putText "[ce_listener] receiving"
      r <- recv s
      C.putStrLn r
      case Aeson.decode (BSL.fromStrict r) of
        Nothing -> putText "[ce_listener] unknown message from central exchange"
        Just ceMessage -> do
          -- subs <- atomically $ readTVar subscriptions
          filteredSubs <- atomically $ do
            let subs' = UnfoldlM.filter (pure . ceMessageSubPred ceMessage) $ SSet.unfoldlM subscriptions
            UnfoldlM.foldlM' (\acc sub -> pure $ acc <> [sub]) [] subs'
          -- NOTE This isn't safe: we atomically fetch subscriptions,
          -- then send notifications one by one. In the meantime, a
          -- subscription could end or new ones could appear (but is
          -- this really a problem? I new subscription comes up, then
          -- probably they already fetch new tree anyways, and if old
          -- one drops in the meantime, it won't listen to what we
          -- send...)
          -- let filteredSubs = filterCEMessageSubs ceMessage subs
          mapM_ (sendNotification ceMessage) filteredSubs
  where
    sendNotification :: CETypes.CEMessage -> Subscription -> IO ()
    sendNotification ceMessage sub = do
      let ws = s_ws_key_connection sub
      -- send the same thing to everyone for now
      WS.sendDataMessage (wsConn ws) (WS.Text (Aeson.encode $ Notification $ s_topic sub) Nothing)


-- Custom filtering of list of Subscriptions based on
-- CETypes.CEMessage.
-- For example, we can add CEMessage.Broadcast to propagate a
-- notification to all connections.
filterCEMessageSubs :: CETypes.CEMessage -> [Subscription] -> [Subscription]
filterCEMessageSubs ceMessage subscriptions = filter (ceMessageSubPred ceMessage) subscriptions

ceMessageSubPred :: CETypes.CEMessage -> Subscription -> Bool
ceMessageSubPred (CETypes.UpdateTreeFirstLevel node_id) (Subscription { s_topic }) =
  s_topic == UpdateTree node_id
