module Gargantext.Components.Table where

import Control.Monad.Cont.Trans (lift)
import Data.Array (filter)
import Data.Maybe (Maybe(..), maybe)
import Data.Either (Either(..))
import Effect (Effect)
import Effect.Aff (Aff)
import React as React
import React (ReactElement, ReactClass, Children, createElement)
import React.DOM (a, b, b', p, i, h3, hr, div, option, select, span, table, tbody, td, text, th, thead, tr)
import React.DOM.Props (className, href, onChange, onClick, scope, selected, value, style)
import Thermite (PerformAction, Render, Spec, modifyState, simpleSpec, createReactSpec, StateCoTransformer)
import Unsafe.Coerce (unsafeCoerce)

import Gargantext.Prelude

type Rows = Array { row    :: Array ReactElement
                  , delete :: Boolean
                  }

type OrderBy = Maybe (OrderByDirection ColumnName)

type LoadRows = { offset :: Int, limit :: Int, orderBy :: OrderBy } -> Aff Rows

newtype ColumnName = ColumnName String

derive instance eqColumnName :: Eq ColumnName

columnName :: ColumnName -> String
columnName (ColumnName c) = c

data OrderByDirection a = ASC a | DESC a

type Props' =
  ( title        :: String
  , colNames     :: Array ColumnName
  , totalRecords :: Int
  , loadRows     :: LoadRows
  )

type Props = Record Props'

type State =
  { rows        :: Maybe Rows
  , currentPage :: Int
  , pageSize    :: PageSizes
  , orderBy     :: OrderBy
--, tree        :: FTree
  }

initialState :: State
initialState =
  { rows         : Nothing
  , currentPage  : 1
  , pageSize     : PS10
  , orderBy      : Nothing
--, tree         : exampleTree
  }

data Action
  = ChangePageSize PageSizes
  | ChangePage     Int
  | ChangeOrderBy  OrderBy

type ChangePageAction = Int -> Effect Unit

-- | Action
-- ChangePageSize
changePageSize :: PageSizes -> State -> State
changePageSize ps td =
  td { pageSize      = ps
     , currentPage   = 1
     }

-- TODO: Not sure this is the right place for this function.
renderTableHeaderLayout :: { title :: String
                           , desc  :: String
                           , query :: String
                           , date  :: String
                           , user  :: String
                           } -> Array ReactElement
renderTableHeaderLayout {title, desc, query, date, user} =
  [ div [className "row"]
    [ div [className "col-md-3"] [ h3 [] [text title] ]
    , div [className "col-md-9"] [ hr [style {height : "2px",backgroundColor : "black"}] ]
    ]
  , div [className "row"] [ div [className "jumbotron1", style {padding : "12px 0px 20px 12px"}]
        [ div [ className "col-md-8 content"]
              [ p [] [ i [className "fa fa-globe"] []
                     , text $ " " <> desc
                     ]
              , p [] [ i [className "fab fa-searchengin"] []
                     , text $ " " <> query
                     ]
              ]
        , div [ className "col-md-4 content"]
              [ p [] [ i [className "fa fa-calendar"] []
                     , text $ " " <> date
                     ]
              , p [] [ i [className "fa fa-user"] []
                     , text $ " " <> user
                     ]
              ]
        ]
    ]
  ]

tableSpec :: Spec State Props Action
tableSpec = simpleSpec performAction render
  where
    modifyStateAndReload :: (State -> State) -> Props -> State -> StateCoTransformer State Unit
    modifyStateAndReload f {loadRows} state = do
      void $ modifyState f
      loadAndSetRows {loadRows} $ f state

    performAction :: PerformAction State Props Action
    performAction (ChangePageSize ps) =
      modifyStateAndReload $ changePageSize ps
    performAction (ChangePage p) =
      modifyStateAndReload $ _ { currentPage = p }
    performAction (ChangeOrderBy mc) =
      modifyStateAndReload $ _ { orderBy = mc }

    renderColHeader :: (OrderBy -> Effect Unit)
                    -> OrderBy
                    -> ColumnName -> ReactElement
    renderColHeader changeOrderBy currentOrderBy c =
      th [scope "col"] [ b' cs ]
      where
        lnk mc = effectLink (changeOrderBy mc)
        cs :: Array ReactElement
        cs =
          case currentOrderBy of
            Just (ASC d)  | c == d -> [lnk (Just (DESC c)) "ASC ",  lnk Nothing (columnName c)]
            Just (DESC d) | c == d -> [lnk (Just (ASC  c)) "DESC ", lnk Nothing (columnName c)]
            _ -> [lnk (Just (ASC c)) (columnName c)]

    render :: Render State Props Action
    render dispatch {title, colNames, totalRecords}
                    {pageSize, currentPage, orderBy, rows} _ =
      let
        ps = pageSizes2Int pageSize
        totalPages = (totalRecords / ps) + min 1 (totalRecords `mod` ps)
      in
      [ div [className "row"]
        [ div [className "col-md-1"] [b [] [text title]]
        , div [className "col-md-2"] [sizeDD pageSize dispatch]
        , div [className "col-md-3"] [textDescription currentPage pageSize totalRecords]
        , div [className "col-md-3"] [pagination (dispatch <<< ChangePage) totalPages currentPage]
              ]
      , table [ className "table"]
        [ thead [className "thead-dark"]
                [tr [] (renderColHeader (dispatch <<< ChangeOrderBy) orderBy <$> colNames)]
        , tbody [] $ map (tr [] <<< map (\c -> td [] [c]) <<< _.row)
                         (maybe [] identity rows)
                      -- TODO display a loading spinner when rows == Nothing
                      -- instead of an empty list of results.
        ]
      ]

loadAndSetRows :: {loadRows :: LoadRows} -> State -> StateCoTransformer State Unit
loadAndSetRows {loadRows} {pageSize, currentPage, orderBy} = do
  let limit = pageSizes2Int pageSize
      offset = limit * (currentPage - 1)
  rows <- lift $ loadRows {offset, limit, orderBy}
  void $ modifyState (_ { rows = Just rows })

tableClass :: ReactClass {children :: Children | Props'}
tableClass =
  React.component "Table"
    (\this -> do
       {state, render} <- spec this
       pure { state, render
            , componentDidMount: do
                {loadRows} <- React.getProps this
                state' <- React.getState this
                dispatcher' this $ loadAndSetRows {loadRows} state'
            })
  where
    { spec, dispatcher' } = createReactSpec tableSpec initialState

tableElt :: Props -> ReactElement
tableElt props = createElement tableClass props []

sizeDD :: PageSizes -> (Action -> Effect Unit) -> ReactElement
sizeDD ps d
  = span []
    [ text "Show : "
    , select [onChange (\e -> d (ChangePageSize $ string2PageSize $ (unsafeCoerce e).target.value))] $ map (optps ps) aryPS
    ]

textDescription :: Int -> PageSizes -> Int -> ReactElement
textDescription currPage pageSize totalRecords
  =  div [className "row1"]
          [ div [className ""]
                [ text $ "Showing " <> show start <> " to " <> show end <> " of " <> show totalRecords ]
          ]
    where
      start = (currPage - 1) * pageSizes2Int pageSize + 1
      end' = currPage * pageSizes2Int pageSize
      end  = if end' > totalRecords then totalRecords else end'

effectLink :: Effect Unit -> String -> ReactElement
effectLink eff msg =
  a [ href "javascript:void()"
    , onClick (const eff)
    ] [text msg]

pagination :: ChangePageAction -> Int -> Int -> ReactElement
pagination changePage tp cp
  = span [] $
    [ text "Pages: ", prev, first, ldots]
    <>
    lnums
    <>
    [b' [text $ " " <> show cp <> " "]]
    <>
    rnums
    <>
    [ rdots, last, next ]
    where
      prev = if cp == 1 then
               text " Previous "
             else
               changePageLink (cp - 1) "Previous"
      next = if cp == tp then
               text " Next "
             else
               changePageLink (cp + 1) "Next"
      first = if cp == 1 then
                text ""
              else
                changePageLink' 1
      last = if cp == tp then
               text ""
             else
               changePageLink' tp
      ldots = if cp >= 5 then
                text " ... "
                else
                text ""
      rdots = if cp + 3 < tp then
                text " ... "
                else
                text ""
      lnums = map changePageLink' $ filter (1  < _) [cp - 2, cp - 1]
      rnums = map changePageLink' $ filter (tp > _) [cp + 1, cp + 2]

      changePageLink :: Int -> String -> ReactElement
      changePageLink i s = span []
          [ text " "
          , effectLink (changePage i) s
          , text " "
          ]

      changePageLink' :: Int -> ReactElement
      changePageLink' i = changePageLink i (show i)

data PageSizes = PS10 | PS20 | PS50 | PS100

derive instance eqPageSizes :: Eq PageSizes

instance showPageSize :: Show PageSizes where
  show PS10  = "10"
  show PS20  = "20"
  show PS50  = "50"
  show PS100 = "100"

pageSizes2Int :: PageSizes -> Int
pageSizes2Int PS10  = 10
pageSizes2Int PS20  = 20
pageSizes2Int PS50  = 50
pageSizes2Int PS100 = 100

aryPS :: Array PageSizes
aryPS = [PS10, PS20, PS50, PS100]

string2PageSize :: String -> PageSizes
string2PageSize "10" = PS10
string2PageSize "20" = PS20
string2PageSize "50" = PS50
string2PageSize "100" = PS100
string2PageSize _    = PS10

optps :: PageSizes -> PageSizes -> ReactElement
optps cv val = option [ selected (cv == val), value $ show val ] [text $ show val]
