{-# OPTIONS_GHC -Wno-deprecations #-}
module Gargantext.API.Server.Named.Private where

import Conduit
import Data.ByteString qualified as B
import Data.ByteString.Builder
import Data.ByteString.Char8 qualified as C8
import Data.Conduit.List qualified as CC
import Data.String
import Data.Text qualified as T
import Data.Text.Encoding qualified as TE
import Gargantext.API.Admin.Auth.Types (AuthenticatedUser(..))
import Gargantext.API.Admin.EnvTypes (Env, env_manager)
import Gargantext.API.Context
import Gargantext.API.Count qualified as Count
import Gargantext.API.Errors.Types
import Gargantext.API.Members (members)
import Gargantext.API.Ngrams.List qualified as List
import Gargantext.API.Node
import Gargantext.API.Node qualified as Tree
import Gargantext.API.Node.Contact as Contact
import Gargantext.API.Node.Corpus.Export qualified as CorpusExport
import Gargantext.API.Node.Document.Export (documentExportAPI)
import Gargantext.API.Node.Phylo.Export qualified as PhyloExport
import Gargantext.API.Node.ShareURL ( shareURL )
import Gargantext.API.Prelude
import Gargantext.API.Routes (addCorpusWithForm, addCorpusWithQuery)
import Gargantext.API.Routes.Named.Private qualified as Named
import Gargantext.API.Server.Named.Ngrams
import Gargantext.API.Server.Named.Viz qualified as Viz
import Gargantext.Core.Types.Individu (User(..))
import Gargantext.Database.Admin.Types.Hyperdata
import Gargantext.Database.Prelude (hasConfig)
import Gargantext.Prelude
import Gargantext.Prelude.Config (gc_frame_write_url)
import Network.HTTP.ReverseProxy
import Network.HTTP.Types (hCacheControl, RequestHeaders, Header, HeaderName)
import Network.HTTP.Types.Header (hHost)
import Network.URI as URI
import Network.Wai (Request, rawPathInfo, requestHeaders)
import Servant hiding (Header)
import Servant.Auth.Swagger ()
import Servant.Server.Generic (AsServerT)

---------------------------------------------------------------------
-- | Server declarations

-- TODO-SECURITY admin only: withAdmin
-- Question: How do we mark admins?
serverGargAdminAPI :: Named.GargAdminAPI (AsServerT (GargM Env BackendInternalError))
serverGargAdminAPI = Named.GargAdminAPI
  { rootsEp       = roots
  , adminNodesAPI = nodesAPI
  }


serverPrivateGargAPI'
  :: Env -> AuthenticatedUser -> Named.GargPrivateAPI' (AsServerT (GargM Env BackendInternalError))
serverPrivateGargAPI' env authenticatedUser@(AuthenticatedUser userNodeId userId)
       =  Named.GargPrivateAPI'
       { gargAdminAPI      = serverGargAdminAPI
       , nodeEp            = nodeAPI authenticatedUser
       , contextEp         = contextAPI (Proxy :: Proxy HyperdataAny) authenticatedUser
       , corpusNodeAPI     = corpusNodeAPI authenticatedUser
       , corpusNodeNodeAPI = nodeNodeAPI (Proxy :: Proxy HyperdataAny) authenticatedUser
       , corpusExportAPI   = CorpusExport.getCorpus
       , annuaireEp        = annuaireNodeAPI authenticatedUser
       , contactAPI        = contactAPI authenticatedUser
       , tableNgramsAPI    = apiNgramsTableDoc authenticatedUser
       , phyloExportAPI    = PhyloExport.api userNodeId
       , documentExportAPI = documentExportAPI userNodeId
       , countAPI          = Count.countAPI
       , graphAPI          = Viz.graphAPI authenticatedUser userId
       , treeAPI           = Tree.treeAPI authenticatedUser
       , treeFlatAPI       = Tree.treeFlatAPI authenticatedUser
       , membersAPI        = members
       , addWithFormEp     = addCorpusWithForm (RootId userNodeId)
       , addWithQueryEp    = addCorpusWithQuery (RootId userNodeId)
       , listGetAPI        = List.getAPI
       , listJsonAPI       = List.jsonAPI
       , listTsvAPI        = List.tsvAPI
       , shareUrlEp        = shareURL
       , notesProxy        = notesProxyImplementation env
       }

notesProxyImplementation :: Env -> Named.NotesProxy (AsServerT (GargM Env BackendInternalError))
notesProxyImplementation env = Named.NotesProxy $ \_frameId -> forwardServer env

forwardServer :: Env -> ServerT Raw m
forwardServer env =
  Tagged $ waiProxyToSettings forwardRequest proxySettings (env ^. env_manager)
  where
    writeFrameURL :: T.Text
    writeFrameURL = env ^. hasConfig . gc_frame_write_url

    assetPath :: T.Text
    assetPath
      | snd forwardedHost /= 80
      = writeFrameURL <> ":" <> T.pack (show $ snd forwardedHost)
      | otherwise
      = writeFrameURL

    removeProxyPath :: ByteString -> ByteString
    removeProxyPath = TE.encodeUtf8
                    . T.replace "/api/v1.0/notes-proxy" ""
                    . TE.decodeUtf8

    proxySettings :: WaiProxySettings
    proxySettings = defaultWaiProxySettings { wpsLogRequest = \req -> traceShow req $ pure ()
                                            , wpsProcessBody =
                                                \_req _res -> Just replaceRelativeLinks
                                            }

    replaceRelativeLinks :: ConduitT B.ByteString (Flush Builder) IO ()
    replaceRelativeLinks = CC.map flushReplace
      where
        -- FIXME(adn) performance.
        flushReplace :: B.ByteString -> Flush Builder
        flushReplace bs = Chunk $ byteString $ traceShowId $ TE.encodeUtf8 . replaceIt . TE.decodeUtf8 $ bs

        replaceIt :: T.Text -> T.Text
        replaceIt t = t & T.replace "src=\"/build/" ("src=\"" <> assetPath <> "/build/")
                        & T.replace "href=\"/build/" ("href=\"" <> assetPath <> "/build/")
                        & T.replace "src=\"/config" ("src=\"" <> assetPath <> "/config")
                        & T.replace "src=\"/js" ("src=\"" <> assetPath <> "/js")

    noCache :: RequestHeaders -> RequestHeaders
    noCache hdrs = (hCacheControl, fromString "no-cache") :
      filter ((/=) hCacheControl . fst) hdrs

    newHost :: RequestHeaders -> RequestHeaders
    newHost hdrs =
      (hHost, fromString $ fst forwardedHost <> ":" <> show (snd forwardedHost)) :
      filter ((/=) hHost . fst) hdrs

    forwardedHost :: (String, Int)
    forwardedHost = fromMaybe (panicTrace "Invalid URI found in the proxied Request.") $ do
      URI{..} <- URI.parseURI (T.unpack writeFrameURL)
      uriAut  <- uriAuthority
      port    <- case uriPort uriAut of
        "" -> pure 80
        xs -> readMaybe (dropWhile (==':') xs)
      pure $ (uriRegName uriAut, port)

    mkProxyDest :: ProxyDest
    mkProxyDest =
      let (hst, prt) = forwardedHost
      in ProxyDest (C8.pack hst) prt

    -- | Forwards the request by substituting back the proxied address into the actual one.
    forwardRequest :: Request -> IO WaiProxyResponse
    forwardRequest originalRequest = do
      let proxiedReq = originalRequest {
                           rawPathInfo = removeProxyPath (rawPathInfo originalRequest)
                         , requestHeaders = (newHost $ noCache $ (requestHeaders originalRequest))
                         }
      pure $ WPRModifiedRequest (traceShowId proxiedReq) $ traceShowId $ mkProxyDest
