module Gargantext.Hooks.UseFeatureFlag where

import Prelude

import Effect.Uncurried (runEffectFn1, EffectFn1)
import Gargantext.Components.CodeEditor (previewPostProcess)
import Gargantext.Utils.Reactix as R2
import Reactix (hooksComponent)
import Reactix as R
import Reactix.DOM.HTML as H

foreign import _useFeatureFlag :: EffectFn1 (Array String) Boolean

here :: R2.Here
here = R2.here "Gargantext.Hooks.UseFeatureFlag"

{-
  useFeatureFlag is a custom React hook that determines whether a feature flag is enabled based on the presence and value of specific cookies. 
  It checks the cookies provided as keys and returns true if any of them are set to 'true'. Otherwise, it returns false.

  This hook is useful for conditional rendering of features based on user-specific feature toggles managed via cookies.
  Usage :

parentCpt :: R.Component ()
parentCpt = here.component "container" cpt
  where
  cpt _ _ = do
    shouldBeVisible <- useFeatureFlag ["keyOne", "keyTwo", "keyThree"]
    pure $ 
      H.div {} [if shouldBeVisible then H.text "I should be visible" else H.text "Im not visible"]
  
-}
useFeatureFlag :: Array String -> R.Hooks Boolean
useFeatureFlag keys =
  R.unsafeHooksEffect $ runEffectFn1 _useFeatureFlag keys

type FeatureProps =
  ( render :: Boolean -> R.Element
  , keys :: Array String
  )

{-
  Use feature component two wrap your component behind a feature flag.
  It might be cleaner to use it instead of directly using the useFeatureFlag hook
  Usage:
  feature { keys: [ "keyOne" ], render: \shouldRender -> if shouldRender then H.text "hello" else H.text "you cant see me" }
-}
feature :: R2.Leaf FeatureProps
feature = R2.renderLeaf "" $
  ( \{ keys, render } _ -> do
      shouldRender <- useFeatureFlag keys
      pure $
        render shouldRender
  )

hideBehind :: forall a. Array String -> R2.Component a -> R2.Component a
hideBehind keys cpt = R2.renderComponent "" \props children -> do
  shouldRender <- useFeatureFlag keys
  pure $
    if shouldRender then cpt props children
    else H.text ""

