module Gargantext.Components.Nodes.Home where

import Data.Array as Array
import DOM.Simple.Console (log)
import Data.Tuple (fst, snd)
import Data.Maybe (Maybe(..))
import Data.Newtype (class Newtype)
import Data.Sequence as Seq
import Effect (Effect)
import Gargantext.Components.Data.Landing (BlockText(..), BlockTexts(..), Button(..), LandingData(..))
import Gargantext.Components.Lang (LandingLang(..))
import Gargantext.Components.Lang.Landing.EnUS as En
import Gargantext.Components.Lang.Landing.FrFR as Fr
import Gargantext.Components.Nodes.Home.Public (renderPublic)
import Gargantext.Ends (Backend(..))
import Gargantext.License (license)
import Gargantext.Prelude -- (Unit, map, pure, unit, void, ($), (<>), (*>))
import Gargantext.Sessions (Sessions(..), unSessions)
import Gargantext.Sessions as Sessions
import Gargantext.Utils.Reactix as R2
import Reactix as R
import Reactix.DOM.HTML as H
import Routing.Hash (setHash)

thisModule = "Gargantext.Components.Nodes.Home"

type Props = ()

newtype State = State
  { userName :: String
  , password :: String
  }

derive instance newtypeState :: Newtype State _

initialState :: State
initialState = State { userName : "", password : "" }

data Action
  = Documentation
  | Enter
  | Login
  | SignUp

performAction :: Action -> Effect Unit
performAction Documentation = pure unit
performAction Enter = void $ setHash "/search"
performAction Login = void $ setHash "/login"
performAction SignUp = pure unit

langLandingData :: LandingLang -> LandingData
langLandingData LL_FR = Fr.landingData
langLandingData LL_EN = En.landingData

------------------------------------------------------------------------

type HomeProps = ( lang      :: LandingLang
                 , publicBackend :: Backend
                 , backend   :: R.State (Maybe Backend)
                 , sessions  :: R2.Reductor Sessions Sessions.Action
                 , visible  :: R.State Boolean
                 )

homeLayout :: Record HomeProps -> R.Element
homeLayout props = R.createElement homeLayoutCpt props []

homeLayoutCpt :: R.Component HomeProps
homeLayoutCpt = R.hooksComponentWithModule thisModule "homeLayout" cpt
  where
    cpt {lang, backend, publicBackend, sessions, visible} _ = do
      let landingData = langLandingData lang
      pure $ H.span {}
           [ H.div { className: "home-title container1" } [ jumboTitle landingData ]
           , H.div { className: "home-research-form container1" } [] -- TODO put research form

           , if Array.length (unSessions $ fst sessions) > 0 
                then H.div { className: "mx-auto container" }
                           [ H.h1 {} [H.text "Welcome! Just watch the tutoriel now"]
                           , H.img { src: "images/Gargantextuel-212x300.jpg"
                                   , id: "funnyimg"
                                   , title: "tutoriel video here"
                                   }
                           ]
                else joinButton

           , H.div { className: "home-public container1" } [ renderPublic { backend
                                                                          , publicBackend
                                                                          , sessions
                                                                          , visible
                                                                          }
                                                           ]

           , H.div {className:"center"}
                   [ H.h1 {} [ -- H.span {className: "fa fa-star-o"} []
                              H.text ""
                             ]
                   ]
           , H.div { className: "home-landing-data container1" } [ blocksRandomText' landingData ]
           , license
           ]
      where
        joinButton = H.div { className:"flex-space-around center" 
                       , paddingTop: "100px"
                       , paddingBottom: "100px"
                       }
                       [ H.button { className: "btn btn-primary my-2"
                                  , on : {click}
                                  , title: "Connect to the server"
                                  } [ H.text "Join"
                                    ]
                       ]
        click _ = log "click!" *> (snd backend) (const $ Just publicBackend)
                               *> (snd visible) (const true)


------------------------------------------------------------------------

blocksRandomText' :: LandingData -> R.Element
blocksRandomText' (LandingData hd) = blocksRandomText hd.blockTexts

blocksRandomText :: BlockTexts -> R.Element
blocksRandomText (BlockTexts bt) =
  H.div { className: "row" } ( map showBlock bt.blocks )
  where
    showBlock :: BlockText -> R.Element
    showBlock (BlockText b) =
      H.div { className: "col-md-4 content" }
      [ H.h3 {}
        [ H.a { href: b.href, title: b.title}
          [ H.i {className: b.icon} []
          , H.text ("   " <> b.titleText) ] ]
        , H.p {} [ H.text b.text ]
        , H.p {} [ docButton b.docButton ] ]

docButton :: Button -> R.Element
docButton (Button b) =
  H.a { className: "btn btn-outline-primary btn-sm spacing-class"
      , href: b.href
      , target: "blank"
      , title: b.title
      } [ H.span { aria: {hidden : true}
                 , className: "glyphicon glyphicon-hand-right"
                 } []
        , H.text b.text
        ]

-- | TODO
-- <img src='logo.png' onmouseover="this.src='gargantextuel.png';" onmouseout="this.src='logo.png';" />
jumboTitle :: LandingData -> R.Element
jumboTitle (LandingData hd) =
  H.div {} [
    H.div { className: "row" } [
       H.div { className: "mx-auto" } [
          H.div { id: "logo-designed" } [
             H.img { src: "images/logo.png"
                   , title: hd.logoTitle
                   }
             ]
          ]
       ]
    ]

imageEnter :: forall t. LandingData -> t -> R.Element
imageEnter (LandingData hd) action =
  H.div {className: "row"}
  [ H.div {className: "col-md-offset-5 col-md-6 content"}
    [ H.img { src: "images/Gargantextuel-212x300.jpg"
            , id: "funnyimg"
            , title: hd.imageTitle
            , action
            }
    ]
  ]

