{-# LANGUAGE DeriveGeneric   #-}
{-# LANGUAGE TemplateHaskell #-}
{-# LANGUAGE TypeOperators   #-}
module Gargantext.API.Routes.Named.Document (
  -- * Routes types
    DocumentsFromWriteNodesAPI(..)
  , DocumentUploadAPI(..)
  , DocumentExportAPI(..)

  -- * API types
  , Params(..)
  , DocumentUpload(..)

  -- * functions and lenses
  , du_title
  , du_sources
  , du_language
  , du_date
  , du_authors
  , du_abstract
  ) where

import Control.Lens
import Data.Aeson
import Data.Swagger hiding (fieldLabelModifier)
import Data.Text (Text)
import GHC.Generics
import Gargantext.API.Admin.Orchestrator.Types
import Gargantext.API.Node.Document.Export.Types
import Gargantext.Core
import Gargantext.Core.Text.List.Social
import Gargantext.Core.Utils.Prefix
import Gargantext.Utils.Servant (ZIP)
import Prelude
import Servant

newtype DocumentExportAPI mode = DocumentExportAPI
  { documentExportAPI ::
      mode :- Summary "Document Export"
           :> "export"
           :> ( "json"
                :> Get '[JSON] (Headers '[Servant.Header "Content-Disposition" Text] DocumentExport)
              :<|> "json.zip"
                :> Get '[ZIP] (Headers '[Servant.Header "Content-Disposition" Text] DocumentExportZIP)
              :<|> "csv"
                :> Get '[PlainText] (Headers '[Servant.Header "Content-Disposition" Text] Text) )
  } deriving Generic

newtype DocumentsFromWriteNodesAPI mode = DocumentsFromWriteNodesAPI
  { docFromWriteNodesEp :: mode :- Summary " Documents from Write nodes."
                                :> AsyncJobs JobLog '[JSON] Params JobLog
  } deriving Generic


newtype DocumentUploadAPI mode = DocumentUploadAPI
  { uploadDocAsyncEp :: mode :- Summary " Document upload"
                             :> "document"
                             :> "upload"
                             :> "async"
                             :> AsyncJobs JobLog '[JSON] DocumentUpload JobLog
  } deriving Generic


data Params = Params
  { id         :: Int
  , paragraphs :: Text
  , lang       :: Lang
  , selection  :: FlowSocialListWith
  }
  deriving (Generic, Show)


data DocumentUpload = DocumentUpload
  { _du_abstract :: Text
  , _du_authors  :: Text
  , _du_sources  :: Text
  , _du_title    :: Text
  , _du_date     :: Text
  , _du_language :: Text
  }
  deriving Generic


--
-- instances
--

instance FromJSON Params where
  parseJSON = genericParseJSON defaultOptions

instance ToJSON Params where
  toJSON = genericToJSON defaultOptions

instance ToSchema Params

$(makeLenses ''DocumentUpload)

instance ToSchema DocumentUpload

instance FromJSON DocumentUpload
  where
    parseJSON = genericParseJSON
            ( defaultOptions { sumEncoding = ObjectWithSingleField
                            , fieldLabelModifier = unCapitalize . dropPrefix "_du_"
                            , omitNothingFields = True
                            }
            )

instance ToJSON DocumentUpload
  where
    toJSON = genericToJSON
           ( defaultOptions { sumEncoding = ObjectWithSingleField
                            , fieldLabelModifier = unCapitalize . dropPrefix "_du_"
                            , omitNothingFields = True
                            }
           )

