{-|
Module      : Gargantext.Core.AsyncUpdates
Description : Asynchronous updates to the frontend
Copyright   : (c) CNRS, 2024-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX
-}

module Gargantext.Core.AsyncUpdates
where

import Gargantext.Core.Types (NodeId, UserId)
import Protolude


{-
I imagine the workflow as follows:
- somewhere in the code (or in the async job worker) we decide to send
  an update message to all interested users
- such an action (UserAction) can be associated with the triggering
  user (but doesn't have to be)
- we compute interested users for given notification
- we broadcast (using our broker) these notifications to all
  interested users 
- the broadcast message is either simple (meaning: hey, we have new
  data, if you want you can send an update request) or we could send
  the changed data already
-}
    

-- | Various update actions
data UpdateAction =
  -- | Update given Servant Job (we currently send a request every second to get job status)
  --  UpdateJob JobID
  -- | Given parent node id, trigger update of the node and its
  --   children (e.g. list is automatically created in a corpus)
  UpdateTree NodeId
  deriving (Eq, Show)

data UserSource =
    USUser UserId
  | USSystem
  deriving (Eq, Show)
    
-- | Action possibly associated with user who triggered it (there can
--   be system actions as well)
data UserAction =
  UserAction UserSource UpdateAction
  deriving (Eq, Show)
    
-- | Represents a notification that goes to a given user. This is
--   directly sent via WebSockets.
data UserNotification =
  UserNotification UserId UserAction
  deriving (Eq, Show)

-- | What we want now is, given a UserAction action, generate all
--   interested users to which the notification will be sent.
--   This function lives in a monad because we have to fetch users
--   from DB.
notificationsForUserAction :: UserAction -> m [ UserNotification ]
notificationsForUserAction = undefined


-- | Stores connection type associated with given user.
--   We probably should set conn = Servant.API.WebSocket.Connection
data ConnectedUser conn =
  ConnectedUser UserId conn


-- | Given a UserNotification and all connected users, send it to
--   interested ones.
sendNotification :: UserNotification -> [ ConnectedUser conn ] -> m ()
sendNotification = undefined
