module Gargantext.Components.Notifications where

import Control.Monad.Except.Trans (runExceptT)
import Data.Array as A
import Data.Either (Either(..))
import Data.Eq.Generic (genericEq)
import Data.FoldableWithIndex (foldlWithIndex, foldMapWithIndex)
import Data.Generic.Rep (class Generic)
import Data.Hashable (class Hashable, hash)
import Data.HashMap as HM
import Data.Maybe (Maybe(..), fromMaybe, isJust)
import Data.Show.Generic (genericShow)
import Data.Traversable (for)
import Data.Tuple (Tuple(..))
import Effect (Effect)
import Effect.Var (($=))
import Effect.Var as Var
import FFI.Simple ((.=), (..))
import Foreign as F
import Gargantext.Sessions.Types (Session(..))
import Gargantext.Utils.Reactix as R2
import Prelude
import Reactix as R
import Simple.JSON as JSON
import Web.Socket.Event.MessageEvent as ME
import WebSocket as WS


here :: R2.Here
here = R2.here "Gargantext.Components.Notifications"


type NodeId = Int
-- Data.UUID.UUID is not Hashable
type UUID = String

data Topic =
  UpdateTree NodeId
derive instance Generic Topic _
instance Eq Topic where eq = genericEq
instance Show Topic where show = genericShow
instance Hashable Topic where
  hash t = hash $ show t
instance JSON.ReadForeign Topic where
  readImpl f = do
    { type: type_ } <- JSON.readImpl f :: F.F { type :: String }
    case type_ of
      "update_tree" -> do
        { node_id } <- JSON.readImpl f :: F.F { node_id :: NodeId }
        pure $ UpdateTree node_id
      s -> F.fail $ F.ErrorAtProperty "type" $ F.ForeignError $ "unkown type: " <> s
instance JSON.WriteForeign Topic where
  writeImpl (UpdateTree node_id) = JSON.writeImpl { "type": "update_tree"
                                                  , node_id }

data WSRequest =
    WSSubscribe Topic
  | WSUnsubscribe Topic
  | WSAuthorize String
  | WSDeauthorize
  | WSPing
  | WSPong
derive instance Generic WSRequest _
instance Eq WSRequest where eq = genericEq
instance JSON.WriteForeign WSRequest where
  writeImpl (WSSubscribe topic) = JSON.writeImpl { request: "subscribe"
                                                 , topic }
  writeImpl (WSUnsubscribe topic) = JSON.writeImpl { request: "unsubscribe"
                                                   , topic }
  writeImpl (WSAuthorize token) = JSON.writeImpl { request: "authorize"
                                                 , token }
  writeImpl WSDeauthorize = JSON.writeImpl { request: "deauthorize" }
  writeImpl WSPing = JSON.writeImpl "ping"
  writeImpl WSPong = JSON.writeImpl "pong"


data Notification =
    Notification Topic
  | Ping
  | Pong
derive instance Generic Notification _
instance Eq Notification where eq = genericEq
instance JSON.ReadForeign Notification where
  readImpl f = do
    let str = JSON.read_ f :: Maybe String
    case str of
      Nothing -> do
        { notification: topic } <- JSON.readImpl f :: F.F { notification :: Topic }
        pure $ Notification topic
      Just "ping" -> pure Ping
      Just "pong" -> pure Pong
      Just s -> F.fail $ F.ErrorAtProperty "_" $ F.ForeignError $ "unkown string: " <> s


type Callback = Unit -> Effect Unit

type CallbacksHM = HM.HashMap UUID Callback

data State =
  State { callbacks :: HM.HashMap Topic CallbacksHM }

emptyState :: State
emptyState = State { callbacks : HM.empty }

insertCallback :: State -> Topic -> UUID -> Callback -> State
insertCallback (State state@{ callbacks }) topic uuid cb =
  State $ state { callbacks = HM.alter alterCallbacksHM topic callbacks }
  where
    alterCallbacksHM :: Maybe CallbacksHM -> Maybe CallbacksHM
    alterCallbacksHM Nothing = Just $ HM.singleton uuid cb
    alterCallbacksHM (Just hm) = Just $ HM.insert uuid cb hm

removeCallback :: State -> Topic -> UUID -> State
removeCallback (State state@{ callbacks }) topic uuid =
  State $ state { callbacks = HM.alter alterCallbacksHM topic callbacks }
  where
    alterCallbacksHM :: Maybe CallbacksHM -> Maybe CallbacksHM
    alterCallbacksHM Nothing = Nothing
    alterCallbacksHM (Just hm) = Just $ HM.delete uuid hm

-- | Execute all callbacks for a given Topic
callTopic :: State -> Topic -> Effect Unit
callTopic (State { callbacks }) topic = do
  here.log2 "[callTopic] topic" topic
  here.log2 "[callTopic] callbacks" (HM.values callbacks)
  here.log2 "[callTopic] topicCallbacks" (HM.values topicCallbacks)
  _ <- for (HM.values topicCallbacks) $ \cb -> do
    cb unit
  pure unit
  where
    topicCallbacks :: CallbacksHM
    topicCallbacks = fromMaybe HM.empty $ HM.lookup topic callbacks


data WSNotification =
  WSNotification { state :: State

                 -- TODO Implement a WS connection
                 , connection :: Maybe WS.Connection
                   
                 -- This calls R.setRef :: R.Ref State -> Effect Unit
                 -- , insertCallback :: Topic -> UUID -> Effect Unit
                 -- This calls R.setRef :: R.Ref State -> Effect Unit
                 -- , removeCallback :: Topic -> UUID -> Effect Unit
                 }

emptyWSNotification :: WSNotification
emptyWSNotification = WSNotification { state      : emptyState
                                     , connection : Nothing }

isConnected :: WSNotification -> Effect Boolean
isConnected (WSNotification { connection: Nothing }) = pure false
isConnected (WSNotification { connection: Just (WS.Connection conn) }) = do
  rs <- Var.get conn.readyState
  pure $ rs == WS.Open

send :: forall a. (JSON.WriteForeign a) => WSNotification -> a -> Effect Unit
send (WSNotification { connection: Nothing }) _ = pure unit
send (WSNotification { connection: Just (WS.Connection conn) }) d =
  conn.send (WS.Message $ JSON.writeJSON d)

alterState :: WSNotification -> (State -> State) -> Effect Unit
alterState (WSNotification ws') stateCb = do
  let state = ws' .. "state"
  -- here.log2 "[alterState] state" state
  void $ pure $ (ws' .= "state") (stateCb state)

allSubscriptions :: WSNotification -> Array (Tuple Topic (Tuple UUID Callback))
allSubscriptions (WSNotification ws') =
  foldMapWithIndex outerFold callbacks
  where
    State { callbacks } = ws' .. "state"
    innerFold :: Topic -> UUID -> Callback -> Array (Tuple Topic (Tuple UUID Callback))
    innerFold topic uuid cb = [Tuple topic (Tuple uuid cb)]
    outerFold :: Topic -> CallbacksHM -> Array (Tuple Topic (Tuple UUID Callback))
    outerFold topic uuidCb = foldMapWithIndex (innerFold topic) uuidCb

data Action =
    InsertCallback Topic UUID Callback
  | RemoveCallback Topic UUID
  | Call Topic

performAction :: WSNotification -> Action -> Effect Unit
performAction ws (InsertCallback topic uuid cb) = do
  let subscription = WSSubscribe topic
  -- WARNING mutable state
  alterState ws (\s -> insertCallback s topic uuid cb)
  connected <- isConnected ws
  if connected
    then
      send ws subscription
    else do
      pure unit
      -- void $ pure $ (ws' .= "state") (insertCallback ws'.state topic uuid cb)
      -- WSNotification $ ws' { state = insertCallback ws'.state topic uuid cb }
performAction ws (RemoveCallback topic uuid) = do
  let subscription = WSUnsubscribe topic
  -- WARNING mutable state
  alterState ws (\s -> removeCallback s topic uuid)
  connected <- isConnected ws
  if connected
    then
      send ws subscription
    else do
      pure unit
      -- void $ pure $ (ws' .= "state") (removeCallback ws'.state topic uuid)
      -- WSNotification $ ws' { state = removeCallback ws'.state topic uuid }
performAction (WSNotification ws') (Call topic) = do
  -- WARNING mutable state
  let state = ws' .. "state"
  here.log2 "[performAction Call] state" state
  callTopic state topic

-- | NOTE This is very ugly. WebSockets events are asychronous. We
-- | should wrap this similar as is done in 0MQ/nanomsg: make an
-- | infinite loop and just wait for an incoming message. This should
-- | reduce (to zero?) the need for SettableVar's etc (which is not
-- | the functional way to do things).
connect :: WSNotification -> String -> (Maybe Session) -> Effect WSNotification
connect ws@(WSNotification { connection: Just _conn }) _ _ = pure ws
connect (WSNotification ws') url session = do
  connection@(WS.Connection conn) <- WS.newWebSocket (WS.URL url) []
  let ws = WSNotification $ ws' { connection = Just connection } 
  conn.onopen $= (\_ -> do
                     -- authorize user first
                     here.log2 "[connect] session" session
                     case session of
                       Just (Session { token }) ->
                         send ws $ WSAuthorize token
                       Nothing -> pure unit
                     -- send pending subscriptions
                     void $ for (allSubscriptions ws) $ \(Tuple topic _) -> do
                       let subscription = WSSubscribe topic
                       here.log2 "[connect] pending subscription" subscription
                       send ws subscription)
  conn.onmessage $= (onmessage ws)

  pure ws

  where
    onmessage ws me = do
      -- WARNING mutable state
      s <- runExceptT $ F.readString (ME.data_ me)
      case s of
        Left err -> do
          here.log2 "[connect] data received is not a string - I was expecting a JSON string!" err
        Right s' -> do
          let parsed = JSON.readJSON s' :: JSON.E Notification
          case parsed of
            Left err -> do
              here.log2 "[connect] I can't parse message" err
            Right (Notification topic) -> do
              here.log2 "[connect] notification" topic
              performAction ws (Call topic)
            -- pings are quiet
            Right Ping -> pure unit
            Right parsed' -> do
              here.log2 "[connect] onmessage, F.readString" parsed'

mkWSNotification :: String -> Effect WSNotification
mkWSNotification url = connect emptyWSNotification url Nothing

-- TODO useLoaderWithUpdate { loader, path, render, topic }
-- topic can be used to bind with a reload state for a component.
-- If a (Notification topic) arrives, we can bump up reload state
