module Gargantext.Components.TopBar where

import Data.Array (reverse)
import Data.Foldable (intercalate)
import Data.Tuple (fst)
import Data.Tuple.Nested ((/\))
import Gargantext.Prelude
import Gargantext.Types (Handed(..))
import Gargantext.Utils.Reactix as R2
import Reactix as R
import Reactix.DOM.HTML as H

thisModule :: String
thisModule = "Gargantext.Components.TopBar"

type TopBarProps = (handed :: R.State Handed)

topBar :: R2.Component TopBarProps
topBar = R.createElement topBarCpt

topBarCpt :: R.Component TopBarProps
topBarCpt = R.hooksComponentWithModule thisModule "topBar" cpt
  where
    cpt { handed } children = do
      pure $ H.div { id: "dafixedtop"
                   , role: "navigation"
                   , className: "navbar navbar-inverse navbar-fixed-top" 
                   }
                   [ H.div { className: "container-fluid" }
                       [ H.div { className: "navbar-inner" } [ logo false (fst handed)]
                       , infoMenu
                       , H.div { className: "navbar-inner" } [ handButton true handed ]
                       , H.div { className: "navbar-inner" } [ smiley     true        ]
                       ]
                   ]
          where
            navHanded t = if xor t (fst handed == LeftHanded) then " navbar-right" else " navbar-left"
 
            handButton t handed = H.ul { title: "If you are Left Handed you can change\n"
                                           <> "the interface by clicking on me. Click\n"
                                           <> "again to come back to previous state."
                                     , className: "nav navbar-nav" <> navHanded t
                                     } [handedChooser { handed } []]

            smiley t = H.ul { title: "Hello! Looking for the tree ?\n"
                                    <> "Just watch on the other side!\n"
                                    <> "Click on the hand again to see it back."
                            , className : "nav navbar-nav" <> navHanded t
                            }
                            [H.li {} [ H.a {} [H.span {className: "fa fa-question-circle-o"} [] ]]
                            ]

            infoMenu = H.div { className: "collapse navbar-collapse"  <> navHanded false}
                             ( [
                                 H.ul { className: "nav navbar-nav" <> navHanded false} []
                               , H.ul { className: "nav navbar-nav" <> navHanded false} [divDropdownLeft {} []]
                               , handButton false handed
                               {-, H.ul { title: "Dark Mode soon here"
                                       , className : "nav navbar-nav"
                                        } [ H.li {} [ H.a {} [ H.span {className : "fa fa-moon"}[]
                                                                 ]
                                                        ]
                                              ]
                                     -}
                               ] <> children 
                             )

            -- sortHanded = if fst handed == LeftHanded then reverse else reverse -- identity
            -- SB.searchBar {session, databases: allDatabases}


logo :: Boolean -> Handed -> R.Element
logo b handed =
  H.a { className, href: "#/" }
  [ H.img { src, title, width: "30", height: "28" }
  ]
  where
    className = "navbar-brand logoSmall" <> navHanded
    src       = "images/logoSmall.png"
    title     = "Back to home."
    navHanded = if xor b (handed == LeftHanded) then " navbar-right" else ""


divDropdownLeft :: R2.Component ()
divDropdownLeft = R.createElement divDropdownLeftCpt

divDropdownLeftCpt :: R.Component ()
divDropdownLeftCpt = R.hooksComponentWithModule thisModule "divDropdownLeft" cpt
  where
    cpt {} _ = do
      show <- R.useState' false

      pure $ H.li { className: "dropdown" } [
          menuButton { element: menuElement, show } []
        , menuElements { elements, show } []
        ]

    menuElement = LiNav { title : "About Gargantext"
                        , href  : "#"
                        , icon  : "fa fa-info-circle"
                        , text  : "Info" }

    elements = [
      [
        LiNav { title : "Quick start, tutorials and methodology"
              , href  : "https://iscpif.fr/gargantext/your-first-map/"
              , icon  : "fa fa-lightbulb-o"
              , text  : "Tutorials"
              }
      , LiNav { title : "Report bug here"
              , href  : "https://www.iscpif.fr/gargantext/feedback-and-bug-reports/"
              , icon  : "glyphicon glyphicon-bullhorn"
              , text  : "Feedback"
              }
      ]
      , -----------------------------------------------------------
      [ LiNav { title : "Chat"
              , href  : "https://chat.iscpif.fr/channel/gargantext"
              , icon  : "fa fa-rocket"
              , text  : "Chat"
              }
      , LiNav { title : "Forums"
              , href  : "https://discourse.iscpif.fr/c/gargantext"
              , icon  : "fa fa-weixin"
              , text  : "Forum"
              }
      ]
      ,------------------------------------------------------------
      [ LiNav { title : "Code documentation"
              , href  : "https://doc.gargantext.org"
              , icon  : "glyphicon glyphicon-book"
              , text  : "Source Code Documentation"
              }
      , LiNav { title : "API documentation"
              , href  : "https://v4.gargantext.org/swagger-ui"
              , icon  : "fa fa-code-fork"
              , text  : "API documentation"
              }
      , LiNav { title : "Source code"
              , href  : "https://gitlab.iscpif.fr/gargantext/haskell-gargantext"
              , icon  : "fa fa-code"
              , text  : "Source Code"
              }
      ]

      ,------------------------------------------------------------
      [ LiNav { title : "More about us (you)"
              , href  : "https://iscpif.fr"
              , icon  : "glyphicon glyphicon-question-sign"
              , text  : "About"
              }
      ]
    ] -- ===========================================================

type MenuButtonProps = (
    element :: LiNav
  , show :: R.State Boolean
  )

menuButton :: R2.Component MenuButtonProps
menuButton = R.createElement menuButtonCpt
  where
    menuButtonCpt :: R.Component MenuButtonProps
    menuButtonCpt = R.hooksComponentWithModule thisModule "menuButton" cpt

    cpt { element: LiNav { title, href, icon, text }, show: (_ /\ setShow) } _ = do
      pure $ H.a { className: "dropdown-toggle navbar-text"
                -- , data: {toggle: "dropdown"}
                , href, title
                , on: { click: \_ -> setShow $ not }
                , role: "button" } [
          H.span { aria: {hidden : true}, className: icon } []
        , H.text (" " <> text)
        ]

-- | Menu in the sidebar, syntactic sugar
type MenuElementsProps = (
    elements :: Array (Array LiNav)
  , show :: R.State Boolean
  )

menuElements :: R2.Component MenuElementsProps
menuElements = R.createElement menuElementsCpt
  where
    menuElementsCpt :: R.Component MenuElementsProps
    menuElementsCpt = R.hooksComponentWithModule thisModule "menuElements" cpt

    cpt { show: false /\ _ } _ = do
      pure $ H.div {} []
    cpt { elements, show: (true /\ setShow) } _ = do
      pure $ H.ul { className: "dropdown-menu"
                  , on: { click: setShow $ const false }
                  , style: { display: "block" } } $ intercalate divider $ map (map liNav) elements
      where
        divider :: Array R.Element
        divider = [H.li {className: "divider"} []]

-- | surgar for target : "blank"
--data LiNav_ = LiNav_ { title  :: String
--                     , href   :: String
--                     , icon   :: String
--                     , text   :: String
--                     , target :: String
--                     }

data LiNav = LiNav { title :: String
                   , href  :: String
                   , icon  :: String
                   , text  :: String
                   }

liNav :: LiNav -> R.Element
liNav (LiNav { title : title'
             , href  : href'
             , icon  : icon'
             , text  : text'
             }
      ) = H.li {} [ H.a { tabIndex: (-1)
                        , target: "blank"
                        , title: title'
                        , href: href'
                        } [ H.span { className: icon' } []
                          , H.text $ " " <> text'
                          ]
                  ]


type HandedChooserProps = (
  handed :: R.State Handed
  )

handedChooser :: R2.Component HandedChooserProps
handedChooser = R.createElement handedChooserCpt

handedChooserCpt :: R.Component HandedChooserProps
handedChooserCpt = R.hooksComponentWithModule thisModule "handedChooser" cpt
  where
    cpt { handed } _ = do
      pure $ H.li {} [
        H.a {} [
          H.span { className: handedClass handed
                 , on: { click: onClick handed } } []
          ]
        ]

    handedClass (LeftHanded  /\ _) = "fa fa-hand-o-left"
    handedClass (RightHanded /\ _) = "fa fa-hand-o-right"

    onClick (_ /\ setHanded) = setHanded $ \h -> case h of
      LeftHanded  -> RightHanded
      RightHanded -> LeftHanded
