{-|
Module      : Gargantext.Core.AsyncUpdates.Dispatcher.Types
Description : Dispatcher (handles websocket connections, accepts message from central exchange)
Copyright   : (c) CNRS, 2017-Present
License     : AGPL + CECILL v3
Maintainer  : team@gargantext.org
Stability   : experimental
Portability : POSIX

https://gitlab.iscpif.fr/gargantext/haskell-gargantext/issues/341

Docs:
https://dev.sub.gargantext.org/#/share/Notes/187918
    
-}

{-# LANGUAGE TypeOperators #-}
{-# OPTIONS_GHC -fno-warn-unused-matches -fno-warn-unused-imports #-}
    
module Gargantext.Core.AsyncUpdates.Dispatcher.Types where

import Control.Concurrent.Async qualified as Async
import Control.Lens (Getter, view)
import Data.Aeson ((.:), (.=))
import Data.Aeson qualified as Aeson
import Data.Aeson.Types (prependFailure, typeMismatch)
import Data.ByteString.Char8 qualified as C
import Data.ByteString.Lazy qualified as BSL
import Data.List (nubBy)
import DeferredFolds.UnfoldlM qualified as UnfoldlM
import Data.UUID.V4 as UUID
import Gargantext.API.Admin.Auth.Types (AuthenticatedUser(_auth_user_id))
import Gargantext.API.Admin.Types (jwtSettings, Settings, jwtSettings)
import Gargantext.API.Prelude (IsGargServer)
import Gargantext.Core.AsyncUpdates.CentralExchange.Types qualified as CETypes
import Gargantext.Core.AsyncUpdates.Constants as AUConstants
import Gargantext.Core.Types (NodeId, UserId)
import Gargantext.Prelude
import GHC.Conc (TVar, newTVarIO, readTVar, writeTVar)
import Nanomsg
import Network.WebSockets qualified as WS
import Protolude.Base (Show(showsPrec))
import Servant
-- import Servant.API.NamedRoutes ((:-))
import Servant.API.WebSocket qualified as WS
import Servant.Auth.Server (verifyJWT)
import Servant.Server.Generic (AsServer, AsServerT)
import StmContainers.Set as SSet



data Topic =
  -- | Update given Servant Job (we currently send a request every
  -- | second to get job status).
  --  UpdateJob JobID
  -- | Given parent node id, trigger update of the node and its
  --   children (e.g. list is automatically created in a corpus)
  UpdateTree NodeId
  deriving (Eq, Show)
instance Hashable Topic where
  hashWithSalt salt (UpdateTree nodeId) = hashWithSalt salt ("update-tree" :: Text, nodeId)
instance FromJSON Topic where
  parseJSON = Aeson.withObject "Topic" $ \o -> do
    type_ <- o .: "type"
    case type_ of
      "update_tree" -> do
        node_id <- o .: "node_id"
        pure $ UpdateTree node_id
      s -> prependFailure "parsing type failed, " (typeMismatch "type" s)
instance ToJSON Topic where
  toJSON (UpdateTree node_id) = Aeson.object [
      "type" .= toJSON ("update_tree" :: Text)
    , "node_id" .= toJSON node_id
    ]
    
data ConnectedUser =
    CUUser UserId
  | CUPublic
  deriving (Eq, Show)
instance Hashable ConnectedUser where
  hashWithSalt salt (CUUser userId) = hashWithSalt salt ("cuuser" :: Text, userId)
  hashWithSalt salt CUPublic = hashWithSalt salt ("cupublic" :: Text)
  
newtype WSKeyConnection = WSKeyConnection (ByteString, WS.Connection)
instance Hashable WSKeyConnection where
  hashWithSalt salt (WSKeyConnection (key, _conn)) = hashWithSalt salt key
instance Eq WSKeyConnection where
  (==) (WSKeyConnection (key1, _conn1)) (WSKeyConnection (key2, _conn2)) = key1 == key2
instance Show WSKeyConnection where
  showsPrec d (WSKeyConnection (key, _conn)) = showsPrec d $ "WSKeyConnection " <> key
showWSKeyConnection :: WSKeyConnection -> Text
showWSKeyConnection ws = "WSKeyConnection " <> show (wsKey ws)
wsKey :: WSKeyConnection -> ByteString
wsKey (WSKeyConnection (key, _conn)) = key
wsConn :: WSKeyConnection -> WS.Connection
wsConn (WSKeyConnection (_key, conn)) = conn
    
data Subscription =
  Subscription {
      s_connected_user    :: ConnectedUser
    , s_ws_key_connection :: WSKeyConnection
    , s_topic             :: Topic }
    deriving (Eq, Show)
instance Hashable Subscription where
  hashWithSalt salt (Subscription { .. }) =
    hashWithSalt salt ( s_connected_user, s_ws_key_connection, s_topic )
subKey :: Subscription -> ByteString
subKey sub = wsKey $ s_ws_key_connection sub


type Token = Text

    
{-
We accept requests for subscription/unsubscription via websocket.

We could instead handle 1 websocket connection per every topic
subscription (e.g. parse headers in WS.PendingConnection. However, WS
by default can handle 65k concurrent connections. With multiple users
having multiple components open, we could exhaust that limit quickly.

Hence, we architect this to have 1 websocket connection per web
browser.
-}
data WSRequest =
    WSSubscribe Topic
  | WSUnsubscribe Topic
  | WSAuthorize Token
  | WSDeauthorize
  deriving (Eq, Show)
instance FromJSON WSRequest where
  parseJSON = Aeson.withObject "WSRequest" $ \o -> do
    request <- o .: "request"
    case request of
      "subscribe" -> do
        topic <- o .: "topic"
        pure $ WSSubscribe topic
      "unsubscribe" -> do
        topic <- o .: "topic"
        pure $ WSUnsubscribe topic
      "authorize" -> do
         token <- o .: "token"
         pure $ WSAuthorize token
      "deauthorize" -> pure $ WSDeauthorize
      s -> prependFailure "parsing request type failed, " (typeMismatch "request" s)
    
data Dispatcher =
 Dispatcher { d_subscriptions :: SSet.Set Subscription
            -- , d_ws_server     :: WSAPI AsServer
            , d_ce_listener   :: ThreadId
            }


class HasDispatcher env where
  hasDispatcher :: Getter env Dispatcher



data Notification =
   Notification Topic
  deriving (Eq, Show)

instance ToJSON Notification where
  toJSON (Notification topic) = Aeson.object [
      "notification" .= toJSON topic
    ]
    
