{-# LANGUAGE ScopedTypeVariables #-}
{-# LANGUAGE TypeFamilies        #-}
{-# LANGUAGE TypeOperators       #-}
{-# LANGUAGE LambdaCase #-}

module Gargantext.API.Auth.PolicyCheck where

import Servant
import Servant.Server.Internal.Delayed
import Servant.Server.Internal.DelayedIO
import Servant.Ekg

import qualified Servant.Swagger as Swagger
--import Data.Proxy
--import Servant.Auth (Cookie)
--import Servant.Auth.Server.Internal.Class
import Prelude
import Gargantext.API.Admin.Auth.Types
import Gargantext.Core.Types
import qualified Network.HTTP.Types as HTTP
import Data.Foldable
import Gargantext.Database.Prelude (DBCmd)

data AccessResult
  = Allow
  | Deny HTTP.Status

instance Semigroup AccessResult where
  Allow <> Allow        = Allow
  Allow <> Deny status  = Deny status
  Deny status <> Allow  = Deny status
  Deny status <> Deny _ = Deny status

instance Monoid AccessResult where
  mempty = Allow

enforce :: Applicative m => HTTP.Status -> Bool -> m AccessResult
enforce errStatus p = pure $ if p then Allow else Deny errStatus

-- | An access policy for gargantext that governs how resources are accessed
-- and who is entitled to see what.
newtype AccessPolicy =
  AccessPolicy { runAccessPolicy :: forall err. [AccessCheck] -> DBCmd err AccessResult }

data AccessCheck
  = AC_node_owner AuthenticatedUser NodeId

check :: Applicative m => AccessCheck -> m AccessResult
check = \case
  AC_node_owner (AuthenticatedUser nodeId) requestedNodeId
    -> enforce HTTP.status403 $ nodeId == requestedNodeId

stockAccessPolicy :: AccessPolicy
stockAccessPolicy = AccessPolicy (foldlM (\acc ac -> mappend acc <$> check ac) Allow)

data PolicyChecked


instance HasServer sub ctx => HasServer (PolicyChecked :> sub) ctx where
  type ServerT (PolicyChecked :> sub) m = AccessPolicy -> ServerT sub m
  hoistServerWithContext _ pc nt s = hoistServerWithContext (Proxy :: Proxy sub) pc nt . s
  route Proxy ctx d = route (Proxy :: Proxy sub) ctx (d `addParameterCheck` getStockAccessPolicy)
    where
      getStockAccessPolicy :: DelayedIO AccessPolicy
      getStockAccessPolicy = DelayedIO $ pure stockAccessPolicy

instance HasLink sub => HasLink (PolicyChecked :> sub) where
    type MkLink (PolicyChecked :> sub) a = MkLink sub a
    toLink f _ = toLink f (Proxy :: Proxy sub)

instance Swagger.HasSwagger sub => Swagger.HasSwagger (PolicyChecked :> sub) where
    toSwagger _ = Swagger.toSwagger (Proxy :: Proxy sub)

instance HasEndpoint sub => HasEndpoint (PolicyChecked :> sub) where
  getEndpoint _ = getEndpoint (Proxy :: Proxy sub)
  enumerateEndpoints _ = enumerateEndpoints (Proxy :: Proxy sub)
