# SPDX-FileCopyrightText: (C) Gargantext Team <team@gargantext.org>
# SPDX-License-Identifier: CC0-1.0
{
  description = "A Nix Flake for Gargantext's Haskell server";
  # To use this Nix flake you may need to enable Nix flake support for your user with:
  #     echo >>~/.config/nix/nix.conf "experimental-features = nix-command flakes"
  # WARNING: be sure that `nix --version` is greater or equal to 2.18,
  # otherwise nix may not support some attributes used in flake.lock.

  # For any input, one can:
  # Update to the latest commit:
  #     nix flake lock --update-input nixpkgs
  # Or to a specific commit (eg. a green one on https://status.nixos.org):
  #     nix flake lock --override-input nixpkgs github:NixOS/nixpkgs/72da83d9515b43550436891f538ff41d68eecc7f
  # Or to a commit (in /etc/nix/registry.json) of a NixOS host:
  #     nix flake lock --override-input nixpkgs flake:nixpkgs
  inputs = {
    haskell-nix.url = "github:input-output-hk/haskell.nix";

    # For trying to hit cache.iog.io one would have
    # to follow haskell.nix's nixpkgs pinned version,
    # but it may be a few months old, so pin it here instead.
    #nixpkgs.follows = "haskell-nix/nixpkgs";
    nixpkgs.url = "github:NixOS/nixpkgs/nixos-24.05";
    haskell-nix.inputs.nixpkgs.follows = "nixpkgs";

    # Convenient Nix Flake utilities, like flake-utils.lib.eachSystem.
    flake-utils.url = "github:numtide/flake-utils";

    # Git pre-commit hooks.
    git-hooks.url = "github:cachix/git-hooks.nix";
    git-hooks.inputs.nixpkgs.follows = "nixpkgs";

    # FIXME: git.iscpif.fr's gitlab's API does not work well,
    # hence fallback to the git+https:// scheme.
    # WARNING: be sure to use git+https:// and not https://
    # otherwise the rev attribute will be missing for the inputMap to work.
    # gargantext-prelude.url = "gitlab:gargantext/haskell-gargantext-prelude?host=git.iscpif.fr";

    accelerate = { url = "git+https://github.com/AccelerateHS/accelerate.git?submodules=1"; flake = false; };
    accelerate-arithmetic = { url = "github:alpmestan/accelerate-arithmetic"; flake = false; };
    accelerate-llvm = { url = "github:AccelerateHS/accelerate-llvm"; flake = false; };
    # WARNING: using the fix-error branch
    accelerate-utility = { url = "git+https://gitlab.iscpif.fr/amestanogullari/accelerate-utility.git?ref=fix-error"; flake = false; };
    boolexpr = { url = "github:boolexpr/boolexpr"; flake = false; };

    coreNLP = { url = "http://nlp.stanford.edu/software/stanford-corenlp-4.5.4.zip"; flake = false; };
    crawlerArxiv = { url = "git+https://gitlab.iscpif.fr/gargantext/crawlers/arxiv-api.git"; flake = false; };
    crawlerHAL = { url = "git+https://gitlab.iscpif.fr/gargantext/crawlers/hal.git"; flake = false; };
    crawlerISTEX = { url = "git+https://gitlab.iscpif.fr/gargantext/crawlers/istex.git"; flake = false; };
    crawlerIsidore = { url = "git+https://gitlab.iscpif.fr/gargantext/crawlers/isidore.git"; flake = false; };
    crawlerPubMed = { url = "git+https://gitlab.iscpif.fr/gargantext/crawlers/pubmed.git"; flake = false; };
    data-time-segment = { url = "github:delanoe/data-time-segment"; flake = false; };
    eigen = { url = "github:chessai/eigen"; flake = false; };
    # WARNING: using the wavewave/ghcHEAD branch
    ekg-json = { url = "github:MercuryTechnologies/ekg-json/wavewave/ghcHEAD"; flake = false; };
    epo-api-client = { url = "git+https://gitlab.iscpif.fr/gargantext/crawlers/epo-proxy-api.git"; flake = false; };
    # WARNING: using the debugNaN branch
    gargantext-graph = { url = "git+https://gitlab.iscpif.fr/gargantext/gargantext-graph.git?ref=debugNaN"; flake = false; };
    # WARNING: unmerged commit
    # See https://gitlab.iscpif.fr/gargantext/haskell-gargantext-prelude/merge_requests/13
    gargantext-prelude = { url = "git+https://gitlab.iscpif.fr/gargantext/haskell-gargantext-prelude.git?ref=303-master-refactoring"; flake = false; };
    haskell-igraph = { url = "git+https://gitlab.iscpif.fr/gargantext/haskell-igraph.git"; flake = false; };
    haskell-throttle = { url = "git+https://gitlab.iscpif.fr/gargantext/haskell-throttle"; flake = false; };
    hlcm = { url = "git+https://gitlab.iscpif.fr/gargantext/hlcm.git"; flake = false; };
    # WARNING: using the alp/imap-static branch
    hmatrix = { url = "github:alpmestan/hmatrix/alp/imap-static"; flake = false; };
    hsinfomap = { url = "git+https://gitlab.iscpif.fr/gargantext/haskell-infomap.git"; flake = false; };
    http-reverse-proxy = { url = "github:adinapoli/http-reverse-proxy"; flake = false; };
    iso639 = { url = "git+https://gitlab.iscpif.fr/gargantext/iso639.git"; flake = false; };
    # WARNING: fork of https://github.com/llvm-hs/llvm-hs
    # using adinapoli/llvm-12-ghc-947-compat branch
    # Tracked in: https://gitlab.iscpif.fr/gargantext/haskell-gargantext/issues/318
    llvm-hs = { url = "github:adinapoli/llvm-hs/adinapoli/llvm-12-ghc-947-compat"; flake = false; };
    nanomsg-haskell = { url = "git+https://github.com/garganscript/nanomsg-haskell"; flake = false; };
    opaleye-textsearch = { url = "git+https://gitlab.iscpif.fr/gargantext/opaleye-textsearch.git?allRefs=1"; flake = false; };
    openalex = { url = "git+https://gitlab.iscpif.fr/gargantext/crawlers/openalex.git"; flake = false; };
    patches-class = { url = "git+https://gitlab.iscpif.fr/gargantext/patches-class.git"; flake = false; };
    patches-map = { url = "github:delanoe/patches-map"; flake = false; };
    rdf4h = { url = "github:robstewart57/rdf4h"; flake = false; };
    # WARNING: using the more-exports branch
    servant-job = { url = "github:adinapoli/servant-job"; flake = false; };
    servant-routes = { url = "github:fpringle/servant-routes"; flake = false; };
    servant-xml-conduit = { url = "git+https://gitlab.iscpif.fr/gargantext/servant-xml-conduit"; flake = false; };
    # WARNING: using the alp/static branch
    sparse-linear = { url = "github:alpmestan/sparse-linear/alp/static"; flake = false; };
    toml-parser = { url = "git+https://github.com/glguy/toml-parser/toml-parser-2.0.1.0"; flake = false; };
  };

  # For printing the available outputs:
  #     $ nix -L flake show --allow-import-from-derivation
  # Note that multiple-systems are enabled hence it can only work
  # without IFD because of https://github.com/NixOS/nix/issues/4265,
  # ie. with a non-null materialized=
  outputs = { self, ... }@inputs:
    let
      supportedSystems = with inputs.flake-utils.lib.system; [
        x86_64-linux
        x86_64-darwin
        aarch64-linux
        aarch64-darwin
      ];
    in
    inputs.flake-utils.lib.eachSystem supportedSystems (system:
      let
        pkgs = import inputs.nixpkgs {
          inherit system;
          config = inputs.haskell-nix.config;
          overlays = [
            inputs.haskell-nix.overlay
            (import nix/nixpkgs/overlays/graphviz.nix)
            (import nix/nixpkgs/overlays/igraph.nix)
            (import nix/nixpkgs/overlays/openblas.nix)
          ];
        };

        # A standard library of Nix expressions.
        lib = inputs.nixpkgs.lib;

        # A library of Nix expressions internal to haskell.nix.
        inherit (pkgs.haskell-nix) haskellLib;

        rawCabalProject = lib.readFile ./cabal.project;

        # haskell.nix's main entry point
        project = pkgs.haskell-nix.cabalProject' [
          ({ config, pkgs, ... }: {
            name = "gargantext";

            # Filter-in input files to avoid unnecessary rebuilds
            # after changing any file tracked in Git that is not actually used by cabalProject.
            src = with lib.fileset; toSource {
              root = ./.;
              fileset = unions [
                ./.clippy.dhall
                ./LICENSE
                ./bench-data
                ./cabal.project
                ./cabal.project.freeze
                ./devops
                ./ekg-assets
                #./gargantext-cors-settings.toml
                ./gargantext.cabal
                ./test-data
                (fileFilter (file: lib.any file.hasExt [ "hs" ]) ./bench)
                (fileFilter (file: lib.any file.hasExt [ "hs" ]) ./bin)
                (fileFilter (file: lib.any file.hasExt [ "hs" ]) ./src)
                (fileFilter (file: lib.any file.hasExt [ "hs" ]) ./test)
              ];
            };

            # By default plan evaluation is done on the build system.
            #evalSystem = "x86_64-linux";

            # Retrieve compiler-nix-name from cabal.project's with-compiler field.
            # Eg. `with-compiler: ghc-9.4.7` becomes "ghc947"
            compiler-nix-name =
              lib.replaceStrings [ "-" "." ] [ "" "" ]
                (lib.head (lib.concatLists (lib.filter (l: l != null)
                  (builtins.map (l: builtins.match "^with-compiler: *(.*)" l)
                    (lib.splitString "\n" rawCabalProject)))));

            # Download GHC from Nixpkgs' binary cache instead of IOHK's
            # which would be done by using: pkgs.haskell-nix.compiler
            # Beware that if any dependency has `build-depends: ghc`
            # then` reinstallableLibGhc = false` is required
            # to avoid missing `genprimopcode`.
            # See https://github.com/input-output-hk/haskell.nix/issues/1809#issuecomment-1358469589
            compilerSelection = pkgs: pkgs.haskell.compiler;

            # Pinning the index-state of Hackage,
            # instead of using the latest known by haskell.nix,
            # removes haskell.nix from interfering too much into the reproducibility.
            # It also enables to materialize the plan-nix.
            index-state = haskellLib.parseIndexState rawCabalProject;

            # Materializing a project means caching the nix files
            # generated from the *.cabal/stack.yaml/package.yaml files.
            # To update:
            #     $ nix run .#update-nix-cache-haskell-nix-materialized
            # It's only checked in ciJobs.
            materialized = if builtins.pathExists nix/cache/haskell.nix/materialized then nix/cache/haskell.nix/materialized else null;

            # Using inputMap for each source-repository-package of cabal.project
            # leverages Nix Flake's inputs to automatically get their rev and sha256 hashes
            # and to check upstreams for updates (using `nix flake update`
            # or `nix flake lock --update-input <input>`).
            inputMap =
              let
                # findCabalFiles (in nix-tools/nix-tools/cabal2nix/Main.hs)
                # always prefers package.yaml over *.cabal,
                # but when the resulting *.cabal file is different
                # than a previously existing one,
                # the build fails with an error like this one:
                #     crawlerIsidore.cabal was modified manually, please use --force to overwrite.
                # Hence just remove this out-of-sync package.yaml.
                removePackageYaml = src: pkgs.symlinkJoin {
                  name = "removePackageYaml-patched";
                  paths = [ src ];
                  postBuild = "rm $out/package.yaml";
                  # Preserve rev for the inputMap
                  passthru.rev = src.rev;
                };
                applyPatches = inputName: patches: pkgs.buildPackages.applyPatches
                  {
                    name = "${inputName}-patched";
                    src = inputs.${inputName};
                    inherit patches;
                  } // { inherit (inputs.${inputName}) rev; };
              in
              {
                "https://github.com/AccelerateHS/accelerate-llvm.git" = inputs.accelerate-llvm;
                "https://github.com/AccelerateHS/accelerate.git" = inputs.accelerate;
                "https://github.com/MercuryTechnologies/ekg-json.git" = inputs.ekg-json;
                "https://github.com/boolexpr/boolexpr.git" = inputs.boolexpr;
                "https://github.com/adinapoli/llvm-hs.git" = inputs.llvm-hs;
                "https://github.com/alpmestan/accelerate-arithmetic.git" = applyPatches "accelerate-arithmetic" [
                  nix/haskell.nix/patches/accelerate-arithmetic/0001-remove-test-using-removed-realBandedGramian.patch
                ];
                "https://github.com/alpmestan/hmatrix.git" = inputs.hmatrix;
                "https://github.com/adinapoli/servant-job.git" = removePackageYaml inputs.servant-job;
                "https://github.com/alpmestan/sparse-linear.git" = inputs.sparse-linear;
                "https://github.com/chessai/eigen.git" = inputs.eigen;
                "https://github.com/delanoe/data-time-segment.git" = inputs.data-time-segment;
                "https://github.com/adinapoli/http-reverse-proxy.git" = inputs.http-reverse-proxy;
                "https://github.com/delanoe/patches-map" = inputs.patches-map;
                "https://gitlab.iscpif.fr/gargantext/opaleye-textsearch.git" = inputs.opaleye-textsearch;
                "https://github.com/robstewart57/rdf4h.git" = inputs.rdf4h;
                "https://github.com/fpringle/servant-routes.git" = inputs.servant-routes;
                "https://gitlab.iscpif.fr/amestanogullari/accelerate-utility.git" = inputs.accelerate-utility;
                "https://gitlab.iscpif.fr/gargantext/crawlers/arxiv-api.git" = inputs.crawlerArxiv;
                "https://gitlab.iscpif.fr/gargantext/crawlers/epo-proxy-api.git" = inputs.epo-api-client;
                "https://gitlab.iscpif.fr/gargantext/crawlers/hal.git" = inputs.crawlerHAL;
                "https://gitlab.iscpif.fr/gargantext/haskell-throttle" = inputs.haskell-throttle;
                "https://gitlab.iscpif.fr/gargantext/crawlers/isidore.git" = removePackageYaml inputs.crawlerIsidore;
                "https://gitlab.iscpif.fr/gargantext/crawlers/istex.git" = removePackageYaml inputs.crawlerISTEX;
                "https://gitlab.iscpif.fr/gargantext/crawlers/openalex.git" = inputs.openalex;
                "https://gitlab.iscpif.fr/gargantext/crawlers/pubmed.git" = inputs.crawlerPubMed;
                "https://gitlab.iscpif.fr/gargantext/gargantext-graph.git" = inputs.gargantext-graph;
                "https://gitlab.iscpif.fr/gargantext/haskell-gargantext-prelude" = removePackageYaml inputs.gargantext-prelude;
                "https://gitlab.iscpif.fr/gargantext/haskell-igraph.git" = inputs.haskell-igraph;
                "https://gitlab.iscpif.fr/gargantext/haskell-infomap.git" = inputs.hsinfomap;
                "https://gitlab.iscpif.fr/gargantext/hlcm.git" = inputs.hlcm;
                "https://gitlab.iscpif.fr/gargantext/iso639.git" = inputs.iso639;
                "https://github.com/garganscript/nanomsg-haskell" = inputs.nanomsg-haskell;
                "https://gitlab.iscpif.fr/gargantext/patches-class.git" = inputs.patches-class;
                "https://gitlab.iscpif.fr/gargantext/servant-xml-conduit.git" = inputs.servant-xml-conduit;

                # TODO tag used in cabal.project
                "https://github.com/glguy/toml-parser/toml-parser-2.0.1.0" = inputs.toml-parser;
              };

            # Default project configuration.
            modules = [
              ({ pkgs, ... }: {
                # Make the closure dependency significantly larger
                # but avoid missing genprimopcode with compilerSelection = p: pkgs.haskell.compiler
                reinstallableLibGhc = false;

                packages.haskell-igraph.components.library = {
                  # The generated plan includes pkgs.igraph, giving access to libigraph.so,
                  # but pkgs.igraph.dev is also needed for igraph.h
                  libs = [ pkgs.igraph.dev ];
                  # Extra include needed because haskell-igraph's cbits
                  # use: #include <igraph.h>
                  # not: #include <igraph/igraph.h>
                  configureFlags = [ "--extra-include-dirs=${pkgs.igraph.dev}/include/igraph" ];
                };

                # Link with OpenBLAS optimized libraries.
                # WARNING: OpenBLAS must only be used by trusted code
                # it is inherently unsuitable for security-conscious applications.
                # See nixpkgs/pkgs/development/libraries/science/math/openblas/default.nix
                packages.hmatrix.flags.openblas = true;
                # Not really necessary because nix builds in a sandbox by default anyway.
                packages.hmatrix.flags.disable-default-paths = true;
              })
            ];

            # Shell configuration shared by the default shell
            # and all shells from the flake.variants.
            shell = {
              # By default haskell.nix does not force cabal-install (by setting CABAL_CONFIG=)
              # to use the packages selected by project.plan-nix and available in `ghc-pkg list`,
              # leaving cabal-install in charge of provisioning Haskell packages,
              # which gives more flexibility when developing.
              #exactDeps = false;
              #allToolDeps = true;

              # haskell.nix provisions (in `ghc-pkg list`)
              # the **dependencies** of the packages selected here,
              # which are also **not** selected here.
              #
              # By default haskell.nix selects all _local_ packages here
              # (packages from both the `packages` and the `source-repository-package` stanzas)
              # which therefore excludes `source-repository-package`s from being provisioned,
              #
              # Note that it means `cabal update` must be run to get an index-state.
              # and be able to download and build missing dependencies
              # that depend on `source-repository-package`s.
              # Eg. gargantext's dependency `hstatistics` depends on `hmatrix`,
              # but hmatrix is a `source-repository-package`
              # hence `hstatistics` is not provisioned by haskell.nix.
              #packages = ps: lib.attrValues (haskellLib.selectLocalPackages ps);

              # Add in this list any development tool needed
              # that is not expected to come from the developers' own system.
              nativeBuildInputs = [
                pkgs.graphviz # for `dot`
                pkgs.haskell.packages.${config.compiler-nix-name}.cabal-install
                pkgs.haskell.packages.${config.compiler-nix-name}.ghcid
                pkgs.haskell.packages.${config.compiler-nix-name}.haskell-language-server
                pkgs.haskell.packages.${config.compiler-nix-name}.hlint
              ];

              shellHook =
                ''
                  export GARGANTEXT_CORENLP_SERVER="nix -L run .#coreNLP"

                  cp -f ${pkgs.buildPackages.writeText "cabal.project.local" ''
                    -- Same fix as in haskell.nix's packages.haskell-igraph.components.library,
                    -- but for cabal-install
                    package haskell-igraph
                        extra-include-dirs: ${pkgs.igraph.dev}/include/igraph
                        extra-lib-dirs: ${lib.concatMapStringsSep " " (p: "${lib.getLib p}/lib") [
                          pkgs.igraph
                          pkgs.openblas
                        ]}
                    -- Enable openblas
                    constraints: hmatrix +openblas
                    package hmatrix
                        flags: +openblas
                        extra-lib-dirs: ${lib.concatMapStringsSep " " (p: "${lib.getLib p}/lib") [ pkgs.openblas ]}
                  ''} cabal.project.local

                  cat >&2 ${pkgs.buildPackages.writeText "shellEnterMessage.txt" ''

                    **Warning**
                      This Nix development shell is not configured to provision
                      `cabal.project`'s `source-repository-package`s and their reverse dependencies,
                      therefore `cabal update` has to be run manually to fetch an `index-state`
                      before `cabal build`.
                  ''}
                '' +
                self.checks.${system}.git-hooks-check.shellHook;

              # When true, builds a Hoogle documentation index of all dependencies,
              # and provides a "hoogle" command to search the index.
              withHoogle = true;
            };

            # Variants to the default project configuration above.
            # They're accessed in the flake's outputs with their name prefixed.
            #     $ nix -L build .#haskell-nix-ghc:gargantext:exe:gargantext-phylo-profile
            # Or via `legacyPackages.${system}.project.projectVariants`:
            #     $ nix -L build .#project.projectVariants.haskell-nix-ghc.components.executables.gargantext-phylo-profile
            flake.variants = {
              # For using profiling versions of Haskell packages:
              #     $ nix develop .#profiling
              profiling = {
                modules = [
                  {
                    # Applies to all packages of the Haskell closure. For instance:
                    #     $ nix eval .#project.hsPkgs.containers.components.library.config.enableProfiling
                    #     false
                    #     $ nix eval .#project.projectVariants.profiling.hsPkgs.containers.components.library.config.enableProfiling
                    #     true
                    enableProfiling = true;
                    enableLibraryProfiling = true;
                  }
                ];
              };

              # For using haskell.nix's GHC:
              #     $ nix -L develop .#haskell-nix-ghc
              #     $ nix -L build .#haskell-nix-ghc:gargantext:exe:gargantext-phylo-profile
              haskell-nix-ghc = {
                compilerSelection = lib.mkForce (pkgs: pkgs.haskell-nix.compiler);
                materialized = lib.mkForce null;
                modules = [
                  {
                    # Revert to the default
                    reinstallableLibGhc = lib.mkForce true;
                  }
                ];
              };
            };

            # Enable coverage report in `ciJobs` and `hydraJobs` flake outputs.
            # For building the coverages:
            #     $ nix -L build .#ciJobs.x86_64-linux.coverage.gargantext
            # Alas, coverage fails to find hpc when using Nixpkgs' GHC:
            # gargantext> no such hpc command
            # So for now the haskell-nix-ghc variant has to be used:
            #     $ nix -L build .#project.projectVariants.haskell-nix-ghc.flake"'".ciJobs.coverage.gargantext
            #     $ firefox result/share/hpc/vanilla/html/
            flake.doCoverage = true;
            # Defaults to haskellLib.selectProjectPackages which select cabal.project's `packages`
            # but rather make all `source-repository-package`s also available in `ciJobs.coverage.*`
            flake.packages = haskellLib.selectLocalPackages;
            # FIXME: haskell.nix uses a `doCoverage = lib.mkDefault true` which causes conflicts.
            flake.coverageProjectModule = {
              modules = [
                {
                  packages =
                    let packageNames = project: builtins.attrNames (config.flake.packages project.hsPkgs); in
                    lib.genAttrs (packageNames config) (_: { doCoverage = true; });
                }
              ];
            };

            # Dead-code analysis
            #     $ nix -L build .#weeder-project-analysis
            #     $ bat result
            # Note that there may be false positives
            # and that some file location may be wrong.
            # weeder = {
            #   packages = ps:
            #     haskellLib.selectProjectPackages ps //
            #     lib.getAttrs [
            #       "crawlerArxiv"
            #       "crawlerHAL"
            #       "crawlerIsidore"
            #       "crawlerPubMed"
            #       "epo-api-client"
            #       "gargantext-graph"
            #       "gargantext-prelude"
            #     ]
            #       ps;
            #   # See https://github.com/ocharles/weeder?tab=readme-ov-file#configuration-options
            #   settings = {
            #     roots = [
            #       "^Main.main$"
            #       # Automatically generated by Cabal
            #       "^Paths_.*"
            #     ];
            #     root-instances = [
            #     ];
            #     # Consider all instances of type classes as roots.
            #     type-class-roots = true;
            #     unused-types = true;
            #   };
            # };

            # Make some variables available to all project modules
            _module.specialArgs = {
              # Use specialArgs to avoid avoid infinite recursion
              # when `inputs` is used in `imports`.
              inherit inputs;
            };
            _module.args = {
              inherit system;
              inherit (pkgs.haskell-nix) haskellLib;
            };
          })

          # project modules
          (import nix/haskell.nix/modules/gargantext.nix)
          #(import nix/haskell.nix/modules/weeder.nix)
        ];

        projectFlake = project.flake { };
      in
      {
        legacyPackages = pkgs // {
          # For exploring the project:
          #     $ nix --extra-experimental-features 'flakes repl-flake' repl .
          #     nix-repl> :lf .
          #     nix-repl> legacyPackages.x86_64-linux.project.<TAB>
          inherit project;
        };

        # For building a component of this project:
        #     $ nix -L build .#gargantext:exe:gargantext-phylo-profile
        packages = projectFlake.packages // {
          #weeder-analysis = project.args.weeder.analysis;
        };

        # For entering the default development shell:
        #     $ nix -L develop
        #     $ cabal build --disable-optimization
        #
        # For entering the development shell variant `profiling`:
        #     $ nix -L develop .#profiling
        #     $ cabal run --enable-profiling gargantext-phylo-profile
        devShells = projectFlake.devShells;

        apps = projectFlake.apps // {
          # For updating nix/cache/haskell.nix/materialized:
          #     $ nix run .#update-nix-cache-haskell-nix-materialized
          # It needs to be updated when cabal.freeze or any other input to the plan-nix changes.
          # It's only OK to use it when the plan-nix does not depend on `system`.
          # See https://github.com/input-output-hk/haskell.nix/blob/master/docs/tutorials/materialization.md#when-is-it-ok-to-materialize
          update-nix-cache-haskell-nix-materialized = inputs.flake-utils.lib.mkApp {
            drv = pkgs.writeShellApplication {
              name = "update-nix-cache-haskell-nix-materialized";
              text = ''
                set -eux
                git diff --exit-code
                ${(project.appendModule { materialized = lib.mkForce null; }).plan-nix.passthru.generateMaterialized} nix/cache/haskell.nix/materialized
                git add --all nix/cache/haskell.nix/materialized
                git commit -m "nix: update nix/cache/haskell.nix/materialized"
              '';
            };
          };

          # Register the default project's toolchain,
          # to prevent nix-collect-garbage from removing them from the Nix store.
          # Note that it does not register the roots of the `projectVariants`.
          update-nix-cache-haskell-nix-gc-roots = inputs.flake-utils.lib.mkApp {
            drv = pkgs.writeShellApplication {
              name = "update-nix-cache-haskell-nix-gc-roots";
              text = ''
                set -eux
                rm -rf nix/cache/haskell.nix/gc-roots
                nix-store --add-root nix/cache/haskell.nix/gc-roots/default --indirect --realise ${project.roots}
                nix-store --add-root nix/cache/haskell.nix/gc-roots/coreNLP --indirect --realise ${self.apps.${system}.coreNLP.program}
              '';
            };
          };

          # For garg-test-hspec
          coreNLP =
            # Avoid recompiling openjdk due to any overlay used in the common pkgs.
            let pkgs = import inputs.nixpkgs { inherit system; }; in
            inputs.flake-utils.lib.mkApp {
              drv = pkgs.writeShellApplication {
                name = "coreNLP";
                text = ''
                  set -x
                  exec ${pkgs.openjdk}/bin/java -mx4g -cp '${inputs.coreNLP}/*' edu.stanford.nlp.pipeline.StanfordCoreNLPServer -port 9000 -timeout 15000
                '';
              };
            };
        };

        # For running all checks (very slow):
        #     $ nix -L flake check
        #
        # For building a specific check of the project:
        #     $ nix -L build .#project.hsPkgs.gargantext.components.tests.garg-test-tasty
        #     $ result/bin/garg-test-tasty
        #
        # Alternatively, but slower:
        #     $ nix -L build .#checks.x86_64-linux.gargantext:test:garg-test-tasty
        #     $ bat result/test-stdout
        #
        # See names from:
        #     $ nix -L flake show --allow-import-from-derivation
        # Alas, currently coverage reports do not work (can't find hpc)
        # with nixpkgs.haskellPackages' GHC, so haskell.nix's GHC has to be used:
        #     $ # nix -L build .#project.projectCoverageReport
        #     $ nix -L build .#project.projectVariants.haskell-nix-ghc.projectCoverageReport
        #     $ firefox result/share/hpc/vanilla/html/index.html
        checks = projectFlake.checks // {
          git-hooks-check = inputs.git-hooks.lib.${system}.run {
            src = ./.;
            hooks = {
              #cabal-fmt.enable = true;
              #fourmolu.enable = true;
              #hlint.enable = true;
              nixpkgs-fmt.enable = true;
            };
          };
        };

        # Jobs for the Nix-based continuous integration system: Hydra
        # https://nixos.wiki/wiki/Hydra
        # Note that haskell.nix always set `checkMaterialization = true` in `hydraJobs`.
        #hydraJobs = projectFlake.hydraJobs;

        # `ciJobs` is like `hydraJobs` but with `${system}` first
        # so that the IFDs will not have to run for systems
        # we are not testing (placement of `${system}` is done by `flake-utils.eachSystem`
        # and it treats `hydraJobs` differently from the other flake.
        # Note that haskell.nix always set `checkMaterialization = true` in `ciJobs`.
        ciJobs = projectFlake.ciJobs;
      }
    );

  # Ask users to set Nix config entries in ~/.local/share/nix/trusted-settings.json.
  nixConfig = {
    # This sets the flake to use the IOG nix cache.
    # Only useful when using the haskell-nix-ghc variant.
    extra-substituters = [
      "https://cache.iog.io"
    ];
    extra-trusted-public-keys = [
      "hydra.iohk.io:f/Ea+s+dFdN+3Y/G+FDgSq+a5NEWhJGzdjvKNGv0/EQ="
    ];

    # haskell.nix translates to Nix expressions:
    # - the build plan usually generated in `dist-newstyle/cache/plan.json` by `cabal configure`
    # - and the `.cabal`/`stack.yaml`/`package.yaml` files of projects.
    #
    # haskell.nix can either generate those Nix expressions on-demand
    # by calling its nix-tools' make-install-plan and cabal-to-nix,
    # hence importing them from a derivation (IFD).
    # Or import pre-generated files whenever project's materialized= attribute is not null,
    # and then no longer needs to allow IFD.
    allow-import-from-derivation = "true";
  };
}
