{-# LANGUAGE TypeApplications #-}

module Main where

import Gargantext.Prelude hiding (isInfixOf)

import Control.Monad
import Data.Text (isInfixOf)
import Gargantext.Core.AsyncUpdates.CentralExchange qualified as CE
import Gargantext.Core.AsyncUpdates.Dispatcher qualified as D
import Gargantext.Core.AsyncUpdates.Dispatcher.Types qualified as DT
import Gargantext.Core.Config.Types (NotificationsConfig(..))
import Shelly hiding (FilePath)
import System.Environment (lookupEnv)
import System.IO
import System.Process
import Test.Hspec
import qualified Data.List as List
import qualified Data.Text as T
import qualified Test.API                     as API
import qualified Test.Server.ReverseProxy     as ReverseProxy
import qualified Test.Database.Operations     as DB


startCoreNLPServer :: IO ProcessHandle
startCoreNLPServer = do
  putText "calling start core nlp"
  devNull <- openFile "/dev/null" WriteMode
  -- Make the CoreNLP server configurable by an envvar, eg. set by nix develop
  coreNLPEnv <- lookupEnv "GARGANTEXT_CORENLP_SERVER"
  let (procName, procArgs) = case coreNLPEnv of
                               Just str | name:args <- List.words str -> (name,args)
                               _ -> ("./startServer.sh", [])
  let p = proc procName procArgs
  (_, _, _, hdl) <- (createProcess $ p { cwd = maybe (Just "devops/coreNLP/stanford-corenlp-current") (const Nothing) coreNLPEnv
                    , delegate_ctlc = True
                    , create_group = True
                    , std_out = UseHandle devNull
                    , std_err = UseHandle devNull
                    }) `catch` \e -> case e of
                                           _ | True <- "does not exist" `isInfixOf` (T.pack . show @SomeException $ e)
                                             -> fail $ "Cannot execute "<>procName<>". If this is the " <>
                                                       "first time you are running the tests, you have to run " <>
                                                       "cd devops/coreNLP && ./build.sh first. You have to run it only once, " <>
                                                       "and then you are good to go for the time being."
                                             | otherwise -> throwIO e
  pure hdl

stopCoreNLPServer :: ProcessHandle -> IO ()
stopCoreNLPServer ph = do
  putText "calling stop core nlp"
  interruptProcessGroupOf ph
  putText "calling stop core nlp - done"

withNotifications :: ((NotificationsConfig, ThreadId, D.Dispatcher) -> IO a) -> IO a
withNotifications = bracket startNotifications stopNotifications
  where
    startNotifications :: IO (NotificationsConfig, ThreadId, D.Dispatcher)
    startNotifications = do
      central_exchange <- forkIO $ CE.gServer nc
      dispatcher <- D.newDispatcher nc
      pure (nc, central_exchange, dispatcher)
    stopNotifications :: (NotificationsConfig, ThreadId, D.Dispatcher) -> IO ()
    stopNotifications (_nc, central_exchange, dispatcher) = do
      putText "calling stop notifications"
      killThread central_exchange
      D.terminateDispatcher dispatcher
      putText "calling stop notifications - done"

nc :: NotificationsConfig
nc = NotificationsConfig { _nc_central_exchange_bind = "tcp://*:15560"
                         , _nc_central_exchange_connect = "tcp://localhost:15560"
                         , _nc_dispatcher_bind = "tcp://*:15561"
                         , _nc_dispatcher_connect = "tcp://localhost:15561" }

-- It's especially important to use Hspec for DB tests, because,
-- unlike 'tasty', 'Hspec' has explicit control over parallelism,
-- and it's important that DB tests are run according to a very
-- precise order, as they are not independent from each other.
-- Unfortunately it's not possibly to use the 'tasty-hspec' adapter
-- because by the time we get a 'TestTree' out of the adapter library,
-- the information about parallelism is lost.
--
-- /IMPORTANT/: For these tests to run correctly, you have to run
-- ./devops/coreNLP/build.sh first. You have to run it only /once/,
-- and then you are good to go for the time being.
main :: IO ()
main = do
  hSetBuffering stdout NoBuffering
  -- TODO Ideally remove start/stop notifications and use
  -- Test/API/Setup to initialize this in env
  withNotifications $ \(nc, _, _) -> do
    bracket startCoreNLPServer stopCoreNLPServer $ \_ -> hspec $ do
      API.tests nc
      ReverseProxy.tests
      DB.tests
      DB.nodeStoryTests
      runIO $ putText "tests finished"
  
